import * as React from 'react';
/**
 * Guard method that validates if a shorthand is a slot
 * can be used to extends properties provided by a slot
 *
 * @example
 * ```
 * const backdropSlot = resolveShorthand(backdrop, {
 *  defaultProps: {
 *    onClick: useEventCallback(event => {
 *     if (isResolvedShorthand(backdrop)) {
 *        backdrop.onClick?.(event)
 *      }
 *      // do something after passing click down the line
 *    }),
 *  },
 * })
 * ```
 * @example
 * ```
 * const handleBackDropClick = (event) => {
 *  // do your thing
 * }
 * const backdropSlot = resolveShorthand(backdrop, {
 *  defaultProps: {
 *    onClick: useEventCallback(
 *      mergeCallbacks(isResolvedShorthand(backdrop) ? backdrop.onClick : undefined, handleBackdropClick)
 *    )
 * })
 * ```
 */ export function isResolvedShorthand(shorthand) {
    return shorthand !== null && typeof shorthand === 'object' && !Array.isArray(shorthand) && !React.isValidElement(shorthand);
}
