import * as React_2 from 'react';

/**
 * Creates a slot from a slot shorthand or properties (`props.SLOT_NAME` or `props` itself)
 * @param value - the value of the slot, it can be a slot shorthand, a slot component or a slot properties
 * @param options - values you can pass to alter the signature of a slot, those values are:
 *
 * * `elementType` - the base element type of a slot, defaults to `'div'`
 * * `defaultProps` - similar to a React component declaration, you can provide a slot default properties to be merged with the shorthand/properties provided.
 */
declare function always<Props extends UnknownSlotProps>(value: Props | SlotShorthandValue | undefined, options: SlotOptions<Props>): SlotComponentType<Props>;

/**
 * @internal
 * resolve the trigger props to the children, either by calling the render function, or cloning with the new props.
 */
export declare function applyTriggerPropsToChildren<TriggerChildProps>(children: TriggerProps<TriggerChildProps>['children'], triggerChildProps: TriggerChildProps): React_2.ReactElement | null;

/**
 * Helper type for inferring the type of the as prop from a Props type.
 *
 * For example:
 * ```
 * type Example<T> = T extends AsIntrinsicElement<infer As> ? As : never;
 * ```
 */
declare type AsIntrinsicElement<As extends JSXIntrinsicElementKeys> = {
    as?: As;
};

/**
 * @internal
 * Assertion method to ensure state slots properties are properly declared.
 * A properly declared slot must be declared by using the `slot` method.
 *
 * @example
 * ```tsx
 * export const renderInput_unstable  = (state: InputState): JSXElement => {
 assertSlots<InputSlots>(state);
 return (
 <state.root>
 {state.contentBefore && <state.contentBefore />}
 <state.input />
 {state.contentAfter && <state.contentAfter />}
 </state.root>
 );
 };
 * ```
 */
export declare function assertSlots<Slots extends SlotPropsRecord>(state: unknown): asserts state is SlotComponents<Slots>;

/**
 * Verifies if an application can use DOM.
 */
export declare function canUseDOM(): boolean;

/**
 * @internal
 * Clamps `value` to a number between the min and max.
 *
 * @param value - the value to be clamped
 * @param min - the lowest valid value
 * @param max - the highest valid value
 */
export declare const clamp: (value: number, min: number, max: number) => number;

/**
 * @internal
 * **THIS TYPE IS INTERNAL AND SHOULD NEVER BE EXPOSED**
 */
declare interface ComponentClass<P = {}, S = React_2.ComponentState> extends React_2.StaticLifecycle<P, S> {
    new (props: P): React_2.Component<P, S>;
}

/**
 * Defines the Props type for a component given its slots and the definition of which one is the primary slot,
 * defaulting to root if one is not provided.
 */
export declare type ComponentProps<Slots extends SlotPropsRecord, Primary extends keyof Slots = 'root'> = Omit<Slots, Primary & 'root'> & PropsWithoutRef<ExtractSlotProps<Slots[Primary]>>;

/**
 * Defines the State object of a component given its slots.
 */
export declare type ComponentState<Slots extends SlotPropsRecord> = {
    /**
     * @deprecated
     * The base element type for each slot.
     * This property is deprecated and will be removed in a future version.
     * The slot base element type is declared through `slot.*(slotShorthand, {elementType: ElementType})` instead.
     */
    components: {
        [Key in keyof Slots]-?: React_2.ElementType;
    };
} & {
    [Key in keyof Slots]: ReplaceNullWithUndefined<WithoutSlotRenderFunction<Exclude<Slots[Key], SlotShorthandValue | (Key extends 'root' ? null : never)>>>;
};

/**
 * @internal
 * With react 18, our `children` type starts leaking everywhere and that causes conflicts on component declaration, specially in the `propTypes` property of
 * both `ComponentClass` and `FunctionComponent`.
 *
 * This type substitutes `React.ComponentType` only keeping the function signature, it omits `propTypes`, `displayName` and other properties that are not
 * required for the inference.
 */
declare type ComponentType<P = {}> = ComponentClass<P> | FunctionComponent<P>;

/**
 * @internal
 * @param compare - comparison function for items
 * @returns Priority queue implemented with a min heap
 */
export declare function createPriorityQueue<T>(compare: PriorityQueueCompareFn<T>): PriorityQueue<T>;

/**
 * Helper type that works similar to Omit,
 * but when modifying an union type it will distribute the omission to all the union members.
 *
 * See [distributive conditional types](https://www.typescriptlang.org/docs/handbook/2/conditional-types.html#distributive-conditional-types) for more information
 */
export declare type DistributiveOmit<T, K extends keyof any> = T extends unknown ? Omit<T, K> : T;

/**
 * Similar functionality to `element.contains` DOM API for use without of order DOM elements that
 * checks the virtual parent hierarchy. If a virtual parents exists, it is chosen over the actual parent
 *
 * @internal
 * @returns true if the child can find the parent in its virtual hierarchy
 */
export declare function elementContains(parent: Node | null, child: Node | null): boolean;

/**
 * HTML element types that are not allowed to have children.
 *
 * Reference: https://developer.mozilla.org/en-US/docs/Glossary/Empty_element
 */
declare type EmptyIntrinsicElements = 'area' | 'base' | 'br' | 'col' | 'embed' | 'hr' | 'img' | 'input' | 'link' | 'meta' | 'param' | 'source' | 'track' | 'wbr';

/**
 * Data type for event handlers. It makes data a discriminated union, where each object requires `event` and `type` property.
 * - `event` is the specific event type
 * - `type` is a string literal. It serves as a clear identifier of the event type that reflects the component's state when the event occurred.
 *    For example, the Tree component's `onNavigation` event handler has different `type` for different key presses: `{ event: React.KeyboardEvent<HTMLElement>; type: typeof ArrowRight } | { event: React.KeyboardEvent<HTMLElement>; type: typeof ArrowLeft }`.
 *    Developers can use the `type` property to identify and filter events of interest.
 * See RFC event-handlers-event-type.md for more details.
 *
 * Example usage:
 * type OnOpenChangeData = (
 *   | EventData\<'click', React.MouseEvent\<MyComponentElement\>\>
 *   | EventData\<'keydown', React.KeyboardEvent\<MyComponentElement\>\>
 * ) & \{ open: boolean; \};
 */
export declare type EventData<Type extends string, TEvent> = {
    type: undefined;
    event: React_2.SyntheticEvent | Event;
} | {
    type: Type;
    event: TEvent;
};

/**
 * Type for props that are event handlers.
 * See RFC event-handlers-event-type.md for more details.
 *
 * Example usage:
 * type OnSomeEventData = EventData\<'click', React.MouseEvent\<MyComponentElement\>\> & \{ open: boolean; \};
 * type SomeProps = \{ onSomeEvent?: EventHandler\<OnSomeEventData\>; \};
 */
export declare type EventHandler<TData extends EventData<string, unknown>> = (ev: React_2.SyntheticEvent | Event, data: TData) => void;

declare interface ExoticComponent<P> {
    (props: P): any;
    $$typeof: symbol;
}

/**
 * Removes SlotShorthandValue and null from the slot type, extracting just the slot's Props object.
 */
export declare type ExtractSlotProps<S> = Exclude<S, SlotShorthandValue | null | undefined>;

/**
 * @internal
 * Allows a component to be tagged as a FluentUI trigger component.
 *
 * Triggers are special-case components: they attach event listeners and other props on their child,
 * and use them to trigger another component to show. Examples include `MenuTrigger` and `Tooltip`.
 *
 * A component can be tagged as a trigger as follows:
 * ```ts
 * const MyComponent: React.FC<MyComponentProps> & FluentTriggerComponent = ...;
 *
 * MyComponent.isFluentTriggerComponent = true; // MUST also set this to true
 * ```
 */
export declare type FluentTriggerComponent = {
    isFluentTriggerComponent?: boolean;
};

/**
 * Return type for `React.forwardRef`, including inference of the proper typing for the ref.
 *
 * @remarks
 * {@link React.RefAttributes} is {@link https://github.com/DefinitelyTyped/DefinitelyTyped/discussions/69756 | leaking string references} into `forwardRef` components
 *  after introducing {@link https://github.com/DefinitelyTyped/DefinitelyTyped/pull/68720 | RefAttributes Type Extension}, which shipped in `@types/react@18.2.61`
 * - `forwardRef` component do not support string refs.
 * - uses custom `RefAttributes` which is compatible with all React versions enforcing no `string` allowance.
 */
export declare type ForwardRefComponent<Props> = NamedExoticComponent<Props & RefAttributes<InferredElementRefType<Props>>>;

/**
 * @internal
 *
 * On types/react 18 there are two types being delivered,
 * they rely on the typescript version to decide which will be consumed {@link https://github.com/DefinitelyTyped/DefinitelyTyped/blob/b59dc3ac1e2770fbd6cdbb90ba52abe04c168196/types/react/package.json#L10}
 *
 * If TS is higher than 5.0 then the `FunctionComponent` will be returning ReactNode (which we don't support)
 * If TS is below or equal to 5.0 then the `FunctionComponent` will be returning ReactElement | null (which we support)
 *
 * Since it's not possible to have a single type that works for both cases
 * (as ReactNode is more specific, and this will break while evaluating functions),
 * we need to create our own `FunctionComponent` type
 * that will work for both cases.
 *
 * **THIS TYPE IS INTERNAL AND SHOULD NEVER BE EXPOSED**
 */
declare interface FunctionComponent<P> {
    (props: P): any;
    displayName?: string;
}

/**
 * Returns an object with clientX, clientY for TouchOrMouseEvent.
 * Returns zeros in case the event is not a mouse or a touch event.
 */
export declare function getEventClientCoords(event: TouchOrMouseEvent): {
    clientX: number;
    clientY: number;
};

/**
 * Given an element tagname and user props, filters the props to only allowed props for the given
 * element type.
 *
 * Equivalent to {@link getNativeElementProps}, but more type-safe.
 *
 * @param tagName - The slot's default element type (e.g. 'div')
 * @param props - The component's props object
 * @param excludedPropNames - List of native props to exclude from the returned value
 */
export declare const getIntrinsicElementProps: <Props extends UnknownSlotProps, ExcludedPropKeys extends Extract<keyof Props, string> = never>(tagName: NonNullable<Props["as"]>, props: Props & React_2.RefAttributes<InferredElementRefType<Props>>, excludedPropNames?: ExcludedPropKeys[]) => DistributiveOmit<Props, ExcludedPropKeys | Exclude<keyof Props, "as" | keyof HTMLAttributes>>;

/**
 * Given an element tagname and user props, filters the props to only allowed props for the given
 * element type.
 * @param tagName - Tag name (e.g. "div")
 * @param props - Props object
 * @param excludedPropNames - List of props to disallow
 *
 * @deprecated use getIntrinsicElementProps instead, it is a type-safe version of this method
 */
export declare function getNativeElementProps<TAttributes extends React_2.HTMLAttributes<any>>(tagName: string, props: {}, excludedPropNames?: string[]): TAttributes;

/**
 * Gets the element which is the parent of a given element.
 * This method prefers the virtual parent over real DOM parent when present.
 * @internal
 */
export declare function getParent(child: Node | null, options?: GetParentOptions): Node | null;

declare type GetParentOptions = {
    /**
     * Indicates if getParent() should ignore a virtual parent.
     * @internal
     */
    skipVirtual?: boolean;
};

/**
 * Splits the native props into ones that go to the `root` slot, and ones that go to the primary slot.
 *
 * This function is only for use with components that have a primary slot other than `root`.
 * Most components should use {@link getNativeElementProps} for their root slot if it is the primary slot.
 *
 * @returns An object containing the native props for the `root` and primary slots.
 */
export declare const getPartitionedNativeProps: <Props extends Pick<React_2.HTMLAttributes<HTMLElement>, "style" | "className">, ExcludedPropKeys extends Extract<keyof Props, string> = never>({ primarySlotTagName, props, excludedPropNames, }: {
    /** The primary slot's element type (e.g. 'div') */
    primarySlotTagName: JSXIntrinsicElementKeys;
    /** The component's props object */
    props: Props;
    /** List of native props to exclude from the returned value */
    excludedPropNames?: ExcludedPropKeys[];
}) => {
    root: {
        style: React_2.CSSProperties | undefined;
        className: string | undefined;
    };
    primary: Omit<Props, ExcludedPropKeys>;
};

/**
 * Returns a ref for the React element in a backwards-compatible way.
 *
 * @param element - The element to get the ref for.
 * @returns The ref for the element.
 */
export declare function getReactElementRef<T>(element: React_2.ReactElement | null | undefined): React_2.Ref<T> | undefined;

/**
 * @internal
 * Finds and swaps a provided key for it's right to left format.
 */
export declare const getRTLSafeKey: (key: string, dir: "ltr" | "rtl") => string;

/**
 * Get the className prop set on the slot by the user, without including the default classes added by the component.
 * Custom style hooks should merge this className _after_ any additional classes added by the hook, to ensure that
 * classes added by the user take precedence over the custom style hook.
 *
 * Example usage in a custom style hook:
 * ```ts
 * state.root.className = mergeClasses(
 *   state.root.className,
 *   customStyles.root,
 *   getSlotClassNameProp_unstable(state.root));
 * ```
 *
 * @returns The className prop set on the slot by the user, or undefined if not set.
 */
export declare const getSlotClassNameProp_unstable: (slot: UnknownSlotProps) => string | undefined;

/**
 * Given the state and an array of slot names, will break out `slots` and `slotProps`
 * collections.
 *
 * The root is derived from a mix of `components` props and `as` prop.
 *
 * Slots will render as null if they are rendered as primitives with undefined children.
 *
 * The slotProps will always omit the `as` prop within them, and for slots that are string
 * primitives, the props will be filtered according to the slot type by the type system.
 * For example, if the slot is rendered `as: 'a'`, the props will be filtered for acceptable
 * anchor props. Note that this is only enforced at build time by Typescript -- there is no
 * runtime code filtering props in this function.
 *
 * @deprecated use slot.always or slot.optional combined with assertSlots instead
 *
 * @param state - State including slot definitions
 * @returns An object containing the `slots` map and `slotProps` map.
 */
export declare function getSlots<R extends SlotPropsRecord>(state: unknown): {
    slots: Slots<R>;
    slotProps: ObjectSlotProps<R>;
};

/**
 * Similar to `getSlots`, main difference is that it's compatible with new custom jsx pragma
 *
 * @internal
 * This is an internal temporary method, this method will cease to exist eventually!
 *
 * * ❗️❗️ **DO NOT USE IT EXTERNALLY** ❗️❗️
 *
 * @deprecated use slot.always or slot.optional combined with assertSlots instead
 */
export declare function getSlotsNext<R extends SlotPropsRecord>(state: unknown): {
    slots: Slots<R>;
    slotProps: ObjectSlotProps<R>;
};

/**
 * @internal
 * Gets the trigger element of a FluentTriggerComponent (such as Tooltip or MenuTrigger).
 *
 * In the case where the immediate child is itself a FluentTriggerComponent and/or React Fragment,
 * it returns the first descendant that is _not_ a FluentTriggerComponent or Fragment.
 * This allows multiple triggers to be stacked, and still apply their props to the actual trigger element.
 *
 * For example, the following returns `<div id="child" />`:
 * ```jsx
 * getTriggerChild(
 *   <Tooltip>
 *     <MenuTrigger>
 *       <div id="child" />
 *     </MenuTrigger>
 *   </Tooltip>
 * );
 * ```
 *
 * In the case where the immediate child is not a valid element,
 * null is returned
 */
export declare function getTriggerChild<TriggerChildProps>(children: TriggerProps<TriggerChildProps>['children']): (React_2.ReactElement<Partial<TriggerChildProps>> & {
    ref?: React_2.Ref<any>;
}) | null;

declare type HTMLAttributes = React_2.HTMLAttributes<any>;

declare type HTMLElementConstructorName = 'HTMLElement' | 'HTMLAnchorElement' | 'HTMLAreaElement' | 'HTMLAudioElement' | 'HTMLBaseElement' | 'HTMLBodyElement' | 'HTMLBRElement' | 'HTMLButtonElement' | 'HTMLCanvasElement' | 'HTMLDataElement' | 'HTMLDataListElement' | 'HTMLDetailsElement' | 'HTMLDivElement' | 'HTMLDListElement' | 'HTMLEmbedElement' | 'HTMLFieldSetElement' | 'HTMLFormElement' | 'HTMLHeadingElement' | 'HTMLHeadElement' | 'HTMLHRElement' | 'HTMLHtmlElement' | 'HTMLIFrameElement' | 'HTMLImageElement' | 'HTMLInputElement' | 'HTMLModElement' | 'HTMLLabelElement' | 'HTMLLegendElement' | 'HTMLLIElement' | 'HTMLLinkElement' | 'HTMLMapElement' | 'HTMLMetaElement' | 'HTMLMeterElement' | 'HTMLObjectElement' | 'HTMLOListElement' | 'HTMLOptGroupElement' | 'HTMLOptionElement' | 'HTMLOutputElement' | 'HTMLParagraphElement' | 'HTMLParamElement' | 'HTMLPreElement' | 'HTMLProgressElement' | 'HTMLQuoteElement' | 'HTMLSlotElement' | 'HTMLScriptElement' | 'HTMLSelectElement' | 'HTMLSourceElement' | 'HTMLSpanElement' | 'HTMLStyleElement' | 'HTMLTableElement' | 'HTMLTableColElement' | 'HTMLTableRowElement' | 'HTMLTableSectionElement' | 'HTMLTemplateElement' | 'HTMLTextAreaElement' | 'HTMLTimeElement' | 'HTMLTitleElement' | 'HTMLTrackElement' | 'HTMLUListElement' | 'HTMLVideoElement';

/**
 * Allows to define a prefix that will be used for all IDs generated by useId() hook. It's useful to avoid collisions
 * between different bundles.
 */
export declare const IdPrefixProvider: React_2.Provider<string | undefined>;

/**
 * Infers the element type from props that are declared using ComponentProps.
 */
export declare type InferredElementRefType<Props> = ObscureEventName extends keyof Props ? Required<Props>[ObscureEventName] extends React_2.PointerEventHandler<infer Element> ? Element : never : never;

/**
 * Helper type for {@link Slot}. Modifies `JSXIntrinsicElements<Type>`:
 * * Removes legacy string ref.
 * * Disallows children for empty tags like 'img'.
 */
declare type IntrinsicElementProps<Type extends JSXIntrinsicElementKeys> = Type extends EmptyIntrinsicElements ? PropsWithoutChildren<React_2.PropsWithRef<JSXIntrinsicElement<Type>>> : React_2.PropsWithRef<JSXIntrinsicElement<Type>>;

/**
 * @internal
 * Checks if a given element is a FluentUI trigger (e.g. `MenuTrigger` or `Tooltip`).
 * See the {@link FluentTriggerComponent} type for more info.
 */
export declare function isFluentTrigger(element: React_2.ReactElement): element is React_2.ReactElement<TriggerProps>;

/**
 * Verifies if a given node is an HTMLElement,
 * this method works seamlessly with frames and elements from different documents
 *
 * This is preferred over simply using `instanceof`.
 * Since `instanceof` might be problematic while operating with [multiple realms](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/instanceof#instanceof_and_multiple_realms)
 *
 * @example
 * ```ts
 * isHTMLElement(event.target) && event.target.focus()
 * isHTMLElement(event.target, {constructorName: 'HTMLInputElement'}) && event.target.value // some value
 * ```
 *
 */
export declare function isHTMLElement<ConstructorName extends HTMLElementConstructorName = 'HTMLElement'>(element?: unknown, options?: {
    /**
     * Can be used to provide a custom constructor instead of `HTMLElement`,
     * Like `HTMLInputElement` for example.
     */
    constructorName?: ConstructorName;
}): element is InstanceType<(typeof globalThis)[ConstructorName]>;

/**
 * @internal
 * Checks that the element has default behaviour from user input on click or 'Enter'/'Space' keys
 */
export declare function isInteractiveHTMLElement(element: unknown): boolean;

/**
 * Returns true if event is a mouse event. Useful when sharing logic between touch and mouse interactions.
 */
export declare function isMouseEvent(event: TouchOrMouseEvent): event is MouseEvent | React_2.MouseEvent;

/**
 * Guard method that validates if a shorthand is a slot
 * can be used to extends properties provided by a slot
 *
 * @example
 * ```
 * const backdropSlot = resolveShorthand(backdrop, {
 *  defaultProps: {
 *    onClick: useEventCallback(event => {
 *     if (isResolvedShorthand(backdrop)) {
 *        backdrop.onClick?.(event)
 *      }
 *      // do something after passing click down the line
 *    }),
 *  },
 * })
 * ```
 * @example
 * ```
 * const handleBackDropClick = (event) => {
 *  // do your thing
 * }
 * const backdropSlot = resolveShorthand(backdrop, {
 *  defaultProps: {
 *    onClick: useEventCallback(
 *      mergeCallbacks(isResolvedShorthand(backdrop) ? backdrop.onClick : undefined, handleBackdropClick)
 *    )
 * })
 * ```
 */
export declare function isResolvedShorthand<Shorthand extends Slot<UnknownSlotProps>>(shorthand?: Shorthand): shorthand is ExtractSlotProps<Shorthand>;

/**
 * Evaluates to true if the given type contains exactly one string, or false if it is a union of strings.
 *
 * ```
 * IsSingleton<'a'> // true
 * IsSingleton<'a' | 'b' | 'c'> // false
 * ```
 */
declare type IsSingleton<T extends string> = {
    [K in T]: Exclude<T, K> extends never ? true : false;
}[T];

/**
 * Guard method to ensure a given element is a slot.
 * This is mainly used internally to ensure a slot is being used as a component.
 */
export declare function isSlot<Props extends {}>(element: unknown): element is SlotComponentType<Props>;

/**
 * Returns true if event is a touch event. Useful when sharing logic between touch and mouse interactions.
 */
export declare function isTouchEvent(event: TouchOrMouseEvent): event is TouchEvent | React_2.TouchEvent;

/**
 * Our own alias for `JSX.Element` type that is compatible with both React 17 and React 18+.
 * Use this type when annotating JSX markup in all our code in order to avoid issues between different React versions.
 *
 * Example usage:
 *
 * BAD:
 * ```tsx
 * const renderFoo = (state: FooState) = <div {...props}>Hello World</div>;
 * // infers
 * // R17:  declare const renderFoo: (state: FooState) => JSX.Element;
 * // R18+: declare const renderFoo: (state: FooState) => React.JSX.Element;
 * ```
 *
 * GOOD:
 * ```tsx
 * import type { JSXElement } from '@fluentui/react-utilities';
 * const renderFoo = (state: FooState): JSXElement = <div {...props}>Hello World</div>;
 * ```
 */
export declare type JSXElement = React_2.ReactElement<any, any>;

/**
 * Our own alias for `JSX.IntrinsicElements` type that is compatible with both React 17 and React 18+.
 * Use this type to get the intrinsic elements from React types in order to avoid issues between different React versions.
 */
export declare type JSXIntrinsicElement<Element extends JSXIntrinsicElementKeys> = React_2.ComponentProps<Element>;

/**
 * Type representing all valid JSX intrinsic element names (e.g., 'div', 'button', 'input').
 * It's derived from `React.ElementType` by excluding all custom component types (`React.ComponentType`), ensuring it only includes standard HTML and SVG elements.
 *
 * Use this type when you need to restrict a type to only valid intrinsic element names.
 *
 * @example
 * ```tsx
 * import * as React from 'react';
 * import type { JSXIntrinsicElementKeys } from '@fluentui/react-utilities';
 *
 * const createElement = (tag: JSXIntrinsicElementKeys) => React.createElement(tag, {});
 *
 * createElement('div'); // Valid
 * createElement('span'); // Valid
 * createElement('unknown'); // Error: Argument of type '"unknown"' is not assignable to parameter of type 'JSXIntrinsicElementKeys'.
 * ```
 *
 * This type helps ensure that only valid intrinsic elements are used in scenarios where custom components are not allowed.
 */
export declare type JSXIntrinsicElementKeys = Exclude<React_2.ElementType, React_2.ComponentType>;

/**
 * @internal
 * Combine two event callbacks into a single callback function that calls each one in order.
 *
 * Usage example:
 * ```ts
 * state.slot.onChange = mergeCallbacks(state.slot.onChange, ev => {
 *   // Handle onChange
 * });
 * ```
 *
 * The primary use is to avoid the need to capture an existing callback (`state.slot.onChange` in the example) to a
 * local variable before replacing with a new listener that calls the existing one. This helps avoid bugs like:
 * * Infinite recursion by calling the re-assigned state.slot.onChange if it's not captured to a local variable.
 * * Missing a call to the original onChange due to an early return or other conditional.
 *
 * If you need a callback that is stable between renders, wrap the result in {@link useEventCallback}.
 *
 * @param callback1 - The first callback to be called, or undefined
 * @param callback2 - The second callback to be called, or undefined
 *
 * @returns A function that that calls the provided functions in order
 */
export declare function mergeCallbacks<Args extends unknown[]>(callback1: ((...args: Args) => void) | undefined, callback2: ((...args: Args) => void) | undefined): (...args: Args) => void;

declare interface NamedExoticComponent<P> extends ExoticComponent<P> {
    displayName?: string;
}

export declare type NativeTouchOrMouseEvent = MouseEvent | TouchEvent;

/**
 * @deprecated - use slot.always or slot.optional combined with assertSlots instead
 */
declare type ObjectSlotProps<S extends SlotPropsRecord> = {
    [K in keyof S]-?: any;
};

/**
 * This is part of a hack to infer the element type from a native element *props* type.
 * The only place the original element is found in a native props type (at least that's workable
 * for inference) is in the event handlers, so some of the helper types use this event handler
 * name to infer the original element type.
 *
 * Notes:
 * - Using an extremely obscure event handler reduces the likelihood that its signature will be
 *   modified in any component's props.
 * - Inferring based on a single prop name instead of a larger type like `DOMAttributes<T>` should be
 *   less expensive for typescript to evaluate and is less likely to result in type expansion in .d.ts.
 */
declare type ObscureEventName = 'onLostPointerCaptureCapture';

/**
 * Tiny helper to do the minimal amount of work in duplicating an object but omitting some
 * props. This ends up faster than using object ...rest or reduce to filter.
 *
 * This behaves very much like filteredAssign, but does not merge many objects together,
 * uses an exclusion object map, and avoids spreads all for optimal performance.
 *
 * See perf test for background:
 * https://jsperf.com/omit-vs-rest-vs-reduce/1
 *
 * @param obj - The object to clone
 * @param exclusions - The array of keys to exclude
 */
export declare function omit<TObj extends Record<string, any>, Exclusions extends (keyof TObj)[]>(obj: TObj, exclusions: Exclusions): Omit<TObj, Exclusions[number]>;

export declare type OnSelectionChangeCallback = (event: React_2.SyntheticEvent, selectedItems: Set<SelectionItemId>) => void;

export declare type OnSelectionChangeData = {
    selectedItems: Set<SelectionItemId>;
};

/**
 * Creates a slot from a slot shorthand or properties (`props.SLOT_NAME` or `props` itself)
 * @param value - the value of the slot, it can be a slot shorthand, a slot component or a slot properties
 * @param options - values you can pass to alter the signature of a slot, those values are:
 *
 * * `elementType` - the base element type of a slot, defaults to `'div'`
 * * `defaultProps` - similar to a React component declaration, you can provide a slot default properties to be merged with the shorthand/properties provided
 * * `renderByDefault` - a boolean that indicates if a slot will be rendered even if it's base value is `undefined`.
 * By default if `props.SLOT_NAME` is `undefined` then `state.SLOT_NAME` becomes `undefined`
 * and nothing will be rendered, but if `renderByDefault = true` then `state.SLOT_NAME` becomes an object
 * with the values provided by `options.defaultProps` (or `{}`). This is useful for cases such as providing a default content
 * in case no shorthand is provided, like the case of the `expandIcon` slot for the `AccordionHeader`
 */
declare function optional<Props extends UnknownSlotProps>(value: Props | SlotShorthandValue | undefined | null, options: {
    renderByDefault?: boolean;
} & SlotOptions<Props>): SlotComponentType<Props> | undefined;

/**
 * @internal
 */
export declare interface PriorityQueue<T> {
    all: () => T[];
    clear: () => void;
    contains: (item: T) => boolean;
    dequeue: () => T;
    enqueue: (item: T) => void;
    peek: () => T | null;
    remove: (item: T) => void;
    size: () => number;
}

/**
 * @internal
 */
declare type PriorityQueueCompareFn<T> = (a: T, b: T) => number;

/**
 * Removes the 'children' prop from the given Props type, leaving unions intact (such as the discriminated union created by
 * IntrinsicSlotProps). This allows IntrinsicSlotProps to be used with React.forwardRef.
 */
declare type PropsWithoutChildren<P> = 'children' extends keyof P ? DistributiveOmit<P, 'children'> : P;

/**
 * Removes the 'ref' prop from the given Props type, leaving unions intact (such as the discriminated union created by
 * IntrinsicSlotProps). This allows IntrinsicSlotProps to be used with React.forwardRef.
 */
declare type PropsWithoutRef<P> = 'ref' extends keyof P ? DistributiveOmit<P, 'ref'> : P;

/**
 * @internal
 *
 * on types/react 18 ReactNode becomes a more strict type, which is not compatible with our current implementation. to avoid any issues we are creating our own ReactNode type which allows anything.
 *
 * This type should only be used for inference purposes, and should never be exposed.
 *
 * **THIS TYPE IS INTERNAL AND SHOULD NEVER BE EXPOSED**
 *
 */
declare type ReactNode = any;

export declare type ReactTouchOrMouseEvent = React_2.MouseEvent | React_2.TouchEvent;

/**
 * @internal
 *
 * This type is used to determine if the current version of React is 18+ or not.
 *
 * It checks if the `React.ReactNode` has `{}` it its type.
 * If it is, then it means that the current version of React is lower than 18.
 * If it is not, then it means that the current version of React is 18 or higher.
 * This is useful for ensuring compatibility with different versions of React.
 *
 * **THIS TYPE IS INTERNAL AND SHOULD NEVER BE EXPOSED**
 */
declare type ReactVersionDependent<Modern, Legacy> = {} extends React_2.ReactNode ? Legacy : Modern;

/**
 * This type should be used in place of `React.RefAttributes<T>` in all components that specify `ref` prop.
 *
 * If user is using React 18 types `>=18.2.61`, they will run into type issues of incompatible refs, using this type mitigates this issues across react type versions.
 *
 * @remarks
 *
 * React 18 types introduced Type Expansion Change to the `RefAttributes` interface as patch release.
 * These changes were released in `@types/react@18.2.61` (replacing ref with `LegacyRef`, which leaks `string` into the union type, causing breaking changes between v8/v9 libraries):
 *  - {@link https://github.com/DefinitelyTyped/DefinitelyTyped/pull/68720 | PR }
 *  - {@link https://app.unpkg.com/@types/react@18.2.61/files/index.d.ts | shipped definitions }
 *
 *
 * In React 19 types this was "reverted" back to the original `Ref<T>` type.
 * In order to maintain compatibility with React 17,18,19, we are forced to use our own version of `RefAttributes`.
 *
 */
export declare interface RefAttributes<T> extends React_2.Attributes {
    ref?: React_2.Ref<T> | undefined;
}

/**
 * A Ref function which can be treated like a ref object in that it has an attached
 * current property, which will be updated as the ref is evaluated.
 */
export declare type RefObjectFunction<T> = React_2.RefObject<T | null> & ((value: T | null) => void);

/**
 * @internal
 * If type T includes `null`, remove it and add `undefined` instead.
 */
declare type ReplaceNullWithUndefined<T> = T extends null ? Exclude<T, null> | undefined : T;

/**
 * Resets generated IDs, should be used only in tests.
 */
export declare function resetIdsForTests(): void;

/**
 *
 * Resolves shorthands into slot props, to ensure normalization of the signature
 * being passed down to getSlots method
 * @param value - the base shorthand props
 * @param options - options to resolve shorthand props
 *
 * @deprecated use slot.always, slot.optional, slot.resolveShorthand combined with assertSlots instead
 */
export declare const resolveShorthand: ResolveShorthandFunction<UnknownSlotProps>;

/**
 * Helper function that converts a slot shorthand or properties to a slot properties object
 * The main difference between this function and `slot` is that this function does not return the metadata required for a slot to be considered a properly renderable slot, it only converts the value to a slot properties object
 * @param value - the value of the slot, it can be a slot shorthand or a slot properties object
 */
declare function resolveShorthand_2<Props extends UnknownSlotProps | null | undefined>(value: Props | SlotShorthandValue): Props;

/**
 * @deprecated use slot.always or slot.optional combined with assertSlots instead
 */
export declare type ResolveShorthandFunction<Props extends UnknownSlotProps = UnknownSlotProps> = {
    <P extends Props>(value: P | SlotShorthandValue | undefined, options: ResolveShorthandOptions<P, true>): WithoutSlotRenderFunction<P>;
    <P extends Props>(value: P | SlotShorthandValue | null | undefined, options?: ResolveShorthandOptions<P, boolean>): WithoutSlotRenderFunction<P> | undefined;
};

/**
 * @deprecated - use slot.always or slot.optional combined with assertSlots instead
 */
export declare type ResolveShorthandOptions<Props, Required extends boolean = false> = Required extends true ? {
    required: true;
    defaultProps?: Props;
} : {
    required?: Required;
    defaultProps?: Props;
};

export declare type SelectionHookParams = {
    selectionMode: SelectionMode_2;
    /**
     * Used in uncontrolled mode to set initial selected items on mount
     */
    defaultSelectedItems?: Iterable<SelectionItemId>;
    /**
     * Used to control selected items
     */
    selectedItems?: Iterable<SelectionItemId>;
    /**
     * Called when selection changes
     */
    onSelectionChange?(event: React_2.SyntheticEvent, data: OnSelectionChangeData): void;
};

export declare type SelectionItemId = string | number;

export declare interface SelectionMethods {
    toggleItem(event: React_2.SyntheticEvent, id: SelectionItemId): void;
    selectItem(event: React_2.SyntheticEvent, id: SelectionItemId): void;
    deselectItem(event: React_2.SyntheticEvent, id: SelectionItemId): void;
    clearItems(event: React_2.SyntheticEvent): void;
    isSelected(id: SelectionItemId): boolean;
    toggleAllItems(event: React_2.SyntheticEvent, itemIds: SelectionItemId[]): void;
}

declare type SelectionMode_2 = 'single' | 'multiselect';
export { SelectionMode_2 as SelectionMode }

/**
 * Sets the virtual parent of an element.
 *
 * @internal
 * @param child - Theme element to set the virtual parent
 * @param parent - The virtual parent, use `undefined` to remove a virtual parent relationship
 */
export declare function setVirtualParent(child: Node, parent?: Node): void;

/**
 * The props type and shorthand value for a slot. Type is either a single intrinsic element like `'div'`,
 * or a component like `typeof Button`.
 *
 * If a slot needs to support multiple intrinsic element types, use the `AlternateAs` param (see examples below).
 *
 * By default, slots can be set to `null` to prevent them from being rendered. If a slot must always be rendered,
 * wrap with `NonNullable` (see examples below).
 *
 * @example
 * ```
 * // Intrinsic element examples:
 * Slot<'div'> // Slot is always div
 * Slot<'button', 'a'> // Defaults to button, but allows as="a" with anchor-specific props
 * Slot<'span', 'div' | 'pre'> // Defaults to span, but allows as="div" or as="pre"
 * NonNullable<Slot<'div'>> // Slot that will always be rendered (can't be set to null by the user)
 *
 * // Component examples:
 * Slot<typeof Button> // Slot is always a Button, and accepts all of Button's Props
 * NonNullable<Slot<typeof Label>> // Slot is a Label and will always be rendered (can't be set to null by the user)
 * ```
 */
export declare type Slot<Type extends JSXIntrinsicElementKeys | ComponentType<any> | UnknownSlotProps, AlternateAs extends JSXIntrinsicElementKeys = never> = IsSingleton<Extract<Type, string>> extends true ? WithSlotShorthandValue<Type extends JSXIntrinsicElementKeys ? {
    as?: Type;
} & WithSlotRenderFunction<IntrinsicElementProps<Type>> : Type extends ComponentType<infer Props> ? Props extends UnknownSlotProps ? Props : WithSlotRenderFunction<Props> : Type> | (AlternateAs extends unknown ? {
    as: AlternateAs;
} & WithSlotRenderFunction<IntrinsicElementProps<AlternateAs>> : never) | null : 'Error: First parameter to Slot must not be not a union of types. See documentation of Slot type.';

declare namespace slot {
    export {
        always,
        optional,
        resolveShorthand_2 as resolveShorthand,
        SlotOptions
    }
}
export { slot }

/**
 * @internal
 * Internal cache of the original className prop for the slot, before being modified by the useStyles hook.
 */
export declare const SLOT_CLASS_NAME_PROP_SYMBOL: unique symbol;

/**
 * @internal
 * Internal reference for the render function
 */
export declare const SLOT_ELEMENT_TYPE_SYMBOL: unique symbol;

/**
 * @internal
 * Internal reference for the render function
 */
export declare const SLOT_RENDER_FUNCTION_SYMBOL: unique symbol;

/**
 * Helper type to correctly define the slot class names object.
 */
export declare type SlotClassNames<Slots> = {
    [SlotName in keyof Slots]-?: string;
};

declare type SlotComponents<Slots extends SlotPropsRecord> = {
    [K in keyof Slots]: SlotComponentType<ExtractSlotProps<Slots[K]>> | (null extends Slots[K] ? undefined : never);
};

/**
 * A definition of a slot, as a component, very similar to how a React component is declared,
 * but with some additional metadata that is used to determine how to render the slot.
 */
export declare type SlotComponentType<Props> = WithoutSlotRenderFunction<Props> & FunctionComponent<{
    children?: ReactNode;
}> & {
    /**
     * @internal
     */
    [SLOT_RENDER_FUNCTION_SYMBOL]?: SlotRenderFunction<Props>;
    /**
     * @internal
     */
    [SLOT_ELEMENT_TYPE_SYMBOL]: ComponentType<Props> | (Props extends AsIntrinsicElement<infer As> ? As : JSXIntrinsicElementKeys);
    /**
     * @internal
     * The original className prop for the slot, before being modified by the useStyles hook.
     */
    [SLOT_CLASS_NAME_PROP_SYMBOL]?: string;
};

export declare type SlotOptions<Props extends UnknownSlotProps> = {
    elementType: React_2.ComponentType<Props> | (Props extends AsIntrinsicElement<infer As> ? As : JSXIntrinsicElementKeys);
    defaultProps?: Partial<Props>;
};

/**
 * Matches any component's Slots type (such as ButtonSlots).
 *
 * This should ONLY be used in type templates as in `extends SlotPropsRecord`;
 * it shouldn't be used as a component's Slots type.
 */
export declare type SlotPropsRecord = Record<string, UnknownSlotProps | SlotShorthandValue | null | undefined>;

export declare type SlotRenderFunction<Props> = (Component: React_2.ElementType<Props>, props: Omit<Props, 'as'>) => ReactNode;

/**
 * @deprecated - use slot.always or slot.optional combined with assertSlots instead
 */
export declare type Slots<S extends SlotPropsRecord> = {
    [K in keyof S]: React_2.ElementType<any>;
};

/**
 * The shorthand value of a slot allows specifying its child
 */
export declare type SlotShorthandValue = React_2.ReactElement | string | number | Iterable<ReactNode> | React_2.ReactPortal;

/**
 * When using SSR with Fluent UI, applications must be wrapped in an SSRProvider. This ensures that auto generated ids
 * are consistent between the client and server.
 *
 * @public
 */
export declare const SSRProvider: React_2.FC<{
    children: React_2.ReactNode;
}>;

export declare type TouchOrMouseEvent = NativeTouchOrMouseEvent | ReactTouchOrMouseEvent;

/**
 * A trigger may have a children that could be either:
 * 1. A single element
 * 2. A render function that will receive properties and must return a valid element or null
 * 3. null or undefined
 */
export declare type TriggerProps<TriggerChildProps = unknown> = {
    children?: React_2.ReactElement | ((props: TriggerChildProps) => React_2.ReactElement | null) | null;
};

/**
 * Converts a union type (`A | B | C`) to an intersection type (`A & B & C`)
 */
export declare type UnionToIntersection<U> = (U extends unknown ? (x: U) => U : never) extends (x: infer I) => U ? I : never;

/**
 * Matches any slot props type.
 *
 * This should ONLY be used in type templates as in `extends UnknownSlotProps`;
 * it shouldn't be used as the type of a slot.
 */
export declare type UnknownSlotProps = Pick<React_2.HTMLAttributes<HTMLElement>, 'className' | 'style'> & {
    as?: JSXIntrinsicElementKeys;
    children?: ReactNode;
};

/**
 * @internal
 * Helper to manage a browser requestAnimationFrame.
 * Ensures that the requestAnimationFrame isn't set multiple times at once and is cleaned up
 * when the component is unloaded.
 *
 * @returns A pair of [requestAnimationFrame, cancelAnimationFrame] that are stable between renders.
 */
export declare function useAnimationFrame(): readonly [(fn: FrameRequestCallback) => number, () => void];

/**
 * A React hook that provides a ref for applying the browser's scrollbar width as a CSS property.
 *
 * This hook is SSR-safe and caches measurements per document to avoid redundant calculations.
 * When the ref is attached to an element, the hook automatically applies the measured scrollbar
 * width to the specified CSS property (defaults to 'width').
 *
 * @example
 * ```tsx
 * const scrollbarRef = useApplyScrollbarWidth({ targetDocument: document });
 * return <div ref={scrollbarRef} />;
 * ```
 */
export declare function useApplyScrollbarWidth<T extends HTMLElement>(options?: UseApplyScrollbarWidthOptions): React_2.RefCallback<T>;

declare interface UseApplyScrollbarWidthOptions {
    /**
     * Does not use the cache and recalculates the scrollbar width
     */
    force?: boolean;
    /**
     * CSS property to apply the scrollbar width to.
     * @default 'width'
     */
    property?: string;
}

/**
 * @internal
 *
 * A [`useState`](https://reactjs.org/docs/hooks-reference.html#usestate)-like hook
 * to manage a value that could be either `controlled` or `uncontrolled`,
 * such as a checked state or text input string.
 *
 * @see https://react.dev/learn/sharing-state-between-components#controlled-and-uncontrolled-components for more details on `controlled`/`uncontrolled`
 *
 * @returns an array of the current value and an updater (dispatcher) function.
 * The updater function is referentially stable (won't change during the component's lifecycle).
 * It can take either a new value, or a function which is passed the previous value and returns the new value.
 *
 * ❗️❗️ Calls to the dispatcher will only modify the state if the state is `uncontrolled`.
 * Meaning that if a state is `controlled`, calls to the dispatcher do not modify the state.
 *
 */
export declare const useControllableState: <State>(options: UseControllableStateOptions<State>) => [State, React_2.Dispatch<React_2.SetStateAction<State>>];

/**
 * @internal
 */
export declare type UseControllableStateOptions<State> = {
    /**
     * User-provided default state or initializer, for uncontrolled usage.
     */
    defaultState?: State | (() => State);
    /**
     * User-provided controlled state. `undefined` means internal state will be used.
     */
    state: State | undefined;
    /**
     * Used as the initial state if `state` and `defaultState` are both `undefined`.
     * If `undefined` is the correct initial state, pass that here.
     */
    initialState: State;
};

/**
 * @internal
 * https://reactjs.org/docs/hooks-faq.html#how-to-read-an-often-changing-value-from-usecallback
 *
 * Modified `useCallback` that can be used when dependencies change too frequently. Can occur when
 * e.g. user props are dependencies which could change on every render
 * e.g. volatile values (i.e. useState/useDispatch) are dependencies which could change frequently
 *
 * This should not be used often, but can be a useful re-render optimization since the callback is a ref and
 * will not be invalidated between re-renders
 *
 * @param fn - The callback function that will be used
 */
export declare const useEventCallback: <Args extends unknown[], Return>(fn: (...args: Args) => Return) => ((...args: Args) => Return);

/**
 * @internal
 * Checks if components was mounted the first time.
 * Supports React concurrent/strict mode by using `useEffect`
 * to track the first mount instead of mutating refs during render.
 *
 * @example
 * const isFirstMount = useFirstMount();
 */
export declare function useFirstMount(): boolean;

/**
 * @internal
 * Forces a re-render, similar to `forceUpdate` in class components.
 */
export declare function useForceUpdate(): React_2.DispatchWithoutAction;

/**
 * Hook to generate a unique ID.
 *
 * @param prefix - Optional prefix for the ID. Defaults to 'fui-'.
 * @param providedId - Optional id provided by a parent component. Defaults to the provided value if present,
 *  without conditioning the hook call
 * @returns The ID
 */
export declare function useId(prefix?: string, providedId?: string): string;

/**
 * React currently throws a warning when using useLayoutEffect on the server. To get around it, we can conditionally
 * useEffect on the server (no-op) and useLayoutEffect in the browser. We occasionally need useLayoutEffect to
 * ensure we don't get a render flash for certain operations, but we may also need affected components to render on
 * the server.
 *
 * https://gist.github.com/gaearon/e7d97cdf38a2907924ea12e4ebdf3c85
 * https://github.com/reduxjs/react-redux/blob/master/src/utils/useIsomorphicLayoutEffect.js
 */
export declare const useIsomorphicLayoutEffect: typeof React_2.useEffect;

/**
 * Returns whether the component is currently being server side rendered or hydrated on the client. Can be used to delay
 * browser-specific rendering until after hydration. May cause re-renders on a client when is used within SSRProvider.
 */
export declare function useIsSSR(): boolean;

/**
 * React hook to merge multiple React refs (either MutableRefObjects or ref callbacks) into a single ref callback that
 * updates all provided refs
 * @param refs - Refs to collectively update with one ref value.
 * @returns A function with an attached "current" prop, so that it can be treated like a RefObject.
 */
export declare function useMergedRefs<T>(...refs: (React_2.Ref<T> | undefined)[]): RefObjectFunction<T>;

/**
 * @internal
 */
export declare type UseOnClickOrScrollOutsideOptions = {
    /**
     * The element to listen for the click event
     */
    element: Document | undefined;
    /**
     * Refs to elements that check if the click is outside
     */
    refs: React_2.MutableRefObject<HTMLElement | undefined | null>[];
    /**
     * By default uses element.contains, but custom contain function can be provided
     *
     * @param parent - provided parent element
     * @param child - event target element
     */
    contains?(parent: HTMLElement | null, child: HTMLElement): boolean;
    /**
     * Disables event listeners
     */
    disabled?: boolean;
    /**
     * Disables custom focus event listeners for iframes
     */
    disabledFocusOnIframe?: boolean;
    /**
     * Called if the click is outside the element refs
     */
    callback: (ev: MouseEvent | TouchEvent) => void;
};

/**
 * @internal
 * Utility to perform checks where a click/touch event was made outside a component
 */
export declare const useOnClickOutside: (options: UseOnClickOrScrollOutsideOptions) => void;

/**
 * @internal
 * Utility to perform checks where a click/touch event was made outside a component
 */
export declare const useOnScrollOutside: (options: UseOnClickOrScrollOutsideOptions) => void;

/**
 * @internal
 */
export declare const usePrevious: <ValueType = unknown>(value: ValueType) => ValueType | null;

/**
 * @returns The width in pixels of the scrollbar in the user agent
 * @remarks This hook is not SSR-safe. For SSR-safe scrollbar width application, use the `useApplyScrollbarWidth` from {@link file://./useApplyScrollbarWidth.ts} instead.
 */
export declare function useScrollbarWidth(options: UseScrollbarWidthOptions): number | undefined;

declare interface UseScrollbarWidthOptions {
    /**
     * Reference document to measure the scrollbar width
     */
    targetDocument: Document | null | undefined;
    /**
     * Does not use the cache and recalculates the scrollbar width
     */
    force?: boolean;
}

export declare function useSelection(params: SelectionHookParams): readonly [Set<SelectionItemId>, SelectionMethods];

/**
 * @internal
 * Helper to manage a browser timeout.
 * Ensures that the timeout isn't set multiple times at once and is cleaned up
 * when the component is unloaded.
 *
 * @returns A pair of [setTimeout, clearTimeout] that are stable between renders.
 */
export declare function useTimeout(): readonly [(fn: () => void, delay?: number) => number, () => void];

/**
 * @internal
 */
declare type WithoutSlotRenderFunction<Props> = Props extends unknown ? 'children' extends keyof Props ? Omit<Props, 'children'> & {
    children?: Exclude<Props['children'], Function>;
} : Props : never;

/**
 * @internal
 * Helper type for {@link Slot}. Takes the props we want to support for a slot and adds the ability for `children`
 * to be a render function that takes those props.
 *
 * Notes: For React 17 and earlier, `children` can be a render function that returns a ReactNode.
 * For React 18 and later, `children` can be any value, as React.ReactNode is a more strict type and does not allow functions anymore.
 * This means that the render functions need to be asserted as `SlotRenderFunction<Props>` for React 18 and later.
 *
 * @example
 * ```tsx
 * // For React 17 and earlier:
 * <Component slot={{ children: (Component, props) => <Component {...props} /> }} />
 *
 * // For React 18 and later:
 * <Component slot={{ children: (Component, props) => <Component {...props} /> as SlotRenderFunction<SlotProps> }} />
 * ```
 */
declare type WithSlotRenderFunction<Props> = PropsWithoutChildren<Props> & {
    children?: 'children' extends keyof Props ? ReactVersionDependent<ReactNode, Props['children'] | SlotRenderFunction<Props>> : never;
};

/**
 * Helper type for {@link Slot}. Adds shorthand types that are assignable to the slot's `children`.
 */
declare type WithSlotShorthandValue<Props> = Props | ('children' extends keyof Props ? Extract<SlotShorthandValue, Props['children']> : never);

export { }
