import type { ComponentProps } from '@fluentui/react-utilities';
import type { ComponentState } from '@fluentui/react-utilities';
import type { JSXElement } from '@fluentui/react-utilities';
import type { PortalProps } from '@fluentui/react-portal';
import type { PositioningShorthand } from '@fluentui/react-positioning';
import * as React_2 from 'react';
import type { Slot } from '@fluentui/react-utilities';
import type { SlotClassNames } from '@fluentui/react-utilities';
import type { TriggerProps } from '@fluentui/react-utilities';

/**
 * Data for the Tooltip's onVisibleChange event.
 */
export declare type OnVisibleChangeData = {
    visible: boolean;
    /**
     * The event object, if this visibility change was triggered by a keyboard event on the document element
     * (such as Escape to hide the visible tooltip). Otherwise undefined.
     */
    documentKeyboardEvent?: KeyboardEvent;
};

/**
 * Render the final JSX of Tooltip
 */
export declare const renderTooltip_unstable: (state: TooltipState) => JSXElement;

/**
 * A tooltip provides light weight contextual information on top of its target element.
 */
export declare const Tooltip: React_2.FC<TooltipProps>;

export declare const tooltipClassNames: SlotClassNames<TooltipSlots>;

/**
 * Properties for Tooltip
 */
export declare type TooltipProps = ComponentProps<TooltipSlots> & TriggerProps<TooltipTriggerProps> & Pick<PortalProps, 'mountNode'> & {
    /**
     * The tooltip's visual appearance.
     * * `normal` - Uses the theme's background and text colors.
     * * `inverted` - Higher contrast variant that uses the theme's inverted colors.
     *
     * @default normal
     */
    appearance?: 'normal' | 'inverted';
    /**
     * Delay before the tooltip is hidden, in milliseconds.
     *
     * @default 250
     */
    hideDelay?: number;
    /**
     * Notification when the visibility of the tooltip is changing.
     *
     * **Note**: for backwards compatibility, `event` will be undefined if this was triggered by a keyboard event on
     * the document element. Use `data.documentKeyboardEvent` if the keyboard event object is needed.
     */
    onVisibleChange?: (event: React_2.PointerEvent<HTMLElement> | React_2.FocusEvent<HTMLElement> | undefined, data: OnVisibleChangeData) => void;
    /**
     * Configure the positioning of the tooltip
     *
     * @default above
     */
    positioning?: PositioningShorthand;
    /**
     * (Required) Specifies whether this tooltip is acting as the description or label of its trigger element.
     *
     * * `label` - The tooltip sets the trigger's aria-label or aria-labelledby attribute. This is useful for buttons
     *    displaying only an icon, for example.
     * * `description` - The tooltip sets the trigger's aria-description or aria-describedby attribute.
     * * `inaccessible` - No aria attributes are set on the trigger. This makes the tooltip's content inaccessible to
     *   screen readers, and should only be used if the tooltip's text is available by some other means.
     */
    relationship: 'label' | 'description' | 'inaccessible';
    /**
     * Delay before the tooltip is shown, in milliseconds.
     *
     * @default 250
     */
    showDelay?: number;
    /**
     * Control the tooltip's visibility programatically.
     *
     * This can be used in conjunction with onVisibleChange to modify the tooltip's show and hide behavior.
     *
     * If not provided, the visibility will be controlled by the tooltip itself, based on hover and focus events on the
     * trigger (child) element.
     *
     * @default false
     */
    visible?: boolean;
    /**
     * Render an arrow pointing to the target element
     *
     * @default false
     */
    withArrow?: boolean;
};

/**
 * Slot properties for Tooltip
 */
export declare type TooltipSlots = {
    /**
     * The text or JSX content of the tooltip.
     */
    content: NonNullable<Slot<'div'>>;
};

/**
 * State used in rendering Tooltip
 */
export declare type TooltipState = ComponentState<TooltipSlots> & Pick<TooltipProps, 'mountNode' | 'relationship'> & Required<Pick<TooltipProps, 'appearance' | 'hideDelay' | 'positioning' | 'showDelay' | 'visible' | 'withArrow'>> & {
    children?: JSXElement | null;
    /**
     * Whether the tooltip should be rendered to the DOM.
     */
    shouldRenderTooltip?: boolean;
    /**
     * Ref to the arrow element
     */
    arrowRef?: React_2.Ref<HTMLDivElement>;
    /**
     * CSS class for the arrow element
     */
    arrowClassName?: string;
};

/**
 * The properties that are added to the child of the Tooltip
 */
export declare type TooltipTriggerProps = {
    ref?: React_2.Ref<unknown>;
} & Pick<React_2.HTMLAttributes<HTMLElement>, 'aria-describedby' | 'aria-label' | 'aria-labelledby' | 'onBlur' | 'onFocus' | 'onPointerEnter' | 'onPointerLeave' | 'aria-haspopup' | 'aria-expanded'>;

/**
 * Create the state required to render Tooltip.
 *
 * The returned state can be modified with hooks such as useTooltipStyles_unstable,
 * before being passed to renderTooltip_unstable.
 *
 * @param props - props from this instance of Tooltip
 */
export declare const useTooltip_unstable: (props: TooltipProps) => TooltipState;

/**
 * Apply styling to the Tooltip slots based on the state
 */
export declare const useTooltipStyles_unstable: (state: TooltipState) => TooltipState;

export { }
