import type { ComponentProps } from '@fluentui/react-utilities';
import type { ComponentState } from '@fluentui/react-utilities';
import type { ContextSelector } from '@fluentui/react-context-selector';
import { FC } from 'react';
import type { ForwardRefComponent } from '@fluentui/react-utilities';
import type { JSXElement } from '@fluentui/react-utilities';
import { Provider } from 'react';
import { ProviderProps } from 'react';
import * as React_2 from 'react';
import type { Slot } from '@fluentui/react-utilities';
import { SlotClassNames } from '@fluentui/react-utilities';

export declare type RegisterTabEventHandler = (data: TabRegisterData) => void;

/**
 * Render the final JSX of Tab
 */
export declare const renderTab_unstable: (state: TabState) => JSXElement;

/**
 * Render the final JSX of TabList
 */
export declare const renderTabList_unstable: (state: TabListState, contextValues: TabListContextValues) => JSXElement;

export declare type SelectTabData = {
    /**
     * The value of the selected tab.
     */
    value: TabValue;
};

export declare type SelectTabEvent<E = HTMLElement> = React_2.MouseEvent<E> | React_2.KeyboardEvent<E> | React_2.FocusEvent<E>;

export declare type SelectTabEventHandler = (event: SelectTabEvent, data: SelectTabData) => void;

/**
 * A tab provides a selectable item in a tab list.
 */
export declare const Tab: ForwardRefComponent<TabProps>;

export declare const tabClassNames: SlotClassNames<TabSlots>;

declare type TabInternalSlots = TabSlots & {
    contentReservedSpace?: Slot<'span'>;
};

/**
 * A tab list provides single selection from a set of tabs.
 */
export declare const TabList: ForwardRefComponent<TabListProps>;

export declare const tabListClassNames: SlotClassNames<TabListSlots>;

export declare type TabListContextValue = Pick<TabListProps, 'onTabSelect' | 'selectTabOnFocus' | 'selectedValue' | 'reserveSelectedTabSpace'> & Required<Pick<TabListProps, 'appearance' | 'disabled' | 'size' | 'vertical'>> & {
    /** A callback to allow a tab to register itself with the tab list. */
    onRegister: RegisterTabEventHandler;
    /** A callback to allow a tab to unregister itself with the tab list. */
    onUnregister: RegisterTabEventHandler;
    /**
     * A callback to allow a tab to select itself when pressed.
     */
    onSelect: SelectTabEventHandler;
    /**
     * Gets the registered tab data along with current and previous selected values.
     */
    getRegisteredTabs: () => {
        selectedValue?: TabValue;
        previousSelectedValue?: TabValue;
        registeredTabs: Record<string, TabRegisterData>;
    };
};

/**
 * Context values used in rendering TabList.
 */
export declare type TabListContextValues = {
    /**
     * The context of the tab list available to each tab.
     */
    tabList: TabListContextValue;
};

/**
 * TabList Props
 */
export declare type TabListProps = ComponentProps<TabListSlots> & {
    /**
     * A tab list can supports 'transparent' and 'subtle' appearance.
     *- 'subtle': Minimizes emphasis to blend into the background until hovered or focused.
     *- 'transparent': No background and border styling
     *- 'subtle-circular': Adds background and border styling
     *- 'filled-circular': Adds background styling
     *
     * The appearance affects each of the contained tabs.
     * @default 'transparent'
     */
    appearance?: 'transparent' | 'subtle' | 'subtle-circular' | 'filled-circular';
    /**
     * Tab size may change between unselected and selected states.
     * The default scenario is a selected tab has bold text.
     *
     * When true, this property requests tabs be the same size whether unselected or selected.
     * @default true
     */
    reserveSelectedTabSpace?: boolean;
    /**
     * The value of the tab to be selected by default.
     * Typically useful when the selectedValue is uncontrolled.
     */
    defaultSelectedValue?: TabValue;
    /**
     * A tab list can be set to disable interaction.
     * @default false
     */
    disabled?: boolean;
    /**
     * Raised when a tab is selected.
     */
    onTabSelect?: SelectTabEventHandler;
    /**
     * When true, focusing a tab will select it.
     * @default false
     */
    selectTabOnFocus?: boolean;
    /**
     * The value of the currently selected tab.
     */
    selectedValue?: TabValue;
    /**
     * A tab list can be either 'small', 'medium', or 'large' size.
     * The size affects each of the contained tabs.
     * @default 'medium'
     */
    size?: 'small' | 'medium' | 'large';
    /**
     * A tab list can arrange its tabs vertically.
     * @default false
     */
    vertical?: boolean;
};

export declare const TabListProvider: Provider<TabListContextValue> & FC<ProviderProps<TabListContextValue>>;

export declare type TabListSlots = {
    /**
     * The slot associated with the root element of this tab list.
     */
    root: Slot<'div'>;
};

/**
 * State used in rendering TabList.
 */
export declare type TabListState = ComponentState<Required<TabListSlots>> & TabListContextValue;

/**
 * Tab Props
 */
export declare type TabProps = Omit<ComponentProps<Partial<TabSlots>>, 'content' | 'value'> & Pick<Partial<TabSlots>, 'content'> & {
    /**
     * A tab can be set to disable interaction.
     * @default false
     */
    disabled?: boolean;
    /**
     * The value that identifies this tab when selected.
     */
    value: TabValue;
};

export declare type TabRegisterData = {
    /**
     * The value of the tab.
     */
    value: TabValue;
    /**
     * The reference to the tab HTML element.
     */
    ref: React_2.RefObject<HTMLElement | null>;
};

export declare const tabReservedSpaceClassNames: {
    content: string;
};

export declare type TabSlots = {
    /**
     * Root of the component.
     */
    root: Slot<'button'>;
    /**
     * Icon that renders before the content.
     */
    icon?: Slot<'span'>;
    /**
     * Component children are placed in this slot
     * Avoid using the `children` property in this slot in favour of Component children whenever possible.
     */
    content: NonNullable<Slot<'span'>>;
};

/**
 * State used in rendering Tab
 */
export declare type TabState = ComponentState<TabInternalSlots> & Pick<TabProps, 'value'> & Required<Pick<TabProps, 'disabled'>> & {
    /**
     * A tab supports 'transparent', 'subtle', `subtle-circular` and `filled-circular` appearance.
     */
    appearance?: 'transparent' | 'subtle' | 'subtle-circular' | 'filled-circular';
    /**
     * A tab can have only an icon slot filled and no content.
     */
    iconOnly: boolean;
    /**
     * If this tab is selected
     */
    selected: boolean;
    /**
     * When defined, tab content with selected style is rendered hidden to reserve space.
     * This keeps consistent content size between unselected and selected states.
     *
     * @deprecated - use `contentReservedSpace` internal slot instead.
     */
    contentReservedSpaceClassName?: string;
    /**
     * A tab can be either 'small', 'medium', or 'large' size.
     */
    size: 'small' | 'medium' | 'large';
    /**
     * A tab can arrange its content based on if the tabs in the list are arranged vertically.
     */
    vertical: boolean;
};

/**
 * Any value that identifies a specific tab.
 */
export declare type TabValue = unknown;

/**
 * Create the state required to render Tab.
 *
 * The returned state can be modified with hooks such as useTabStyles_unstable,
 * before being passed to renderTab_unstable.
 *
 * @param props - props from this instance of Tab
 * @param ref - reference to root HTMLElement of Tab
 */
export declare const useTab_unstable: (props: TabProps, ref: React_2.Ref<HTMLElement>) => TabState;

/**
 * Adds additional styling to the active tab selection indicator to create a sliding animation.
 */
export declare const useTabAnimatedIndicatorStyles_unstable: (state: TabState) => TabState;

/**
 * Applies styles to the Tab button slot based on its current state.
 *
 * This hook is typically used internally by `useTabStyles_unstable`. You should
 * only use it directly if you're creating a custom `Tab` component.
 *
 * @param state - The Tab component's current state
 * @param slot - The button slot of the Tab component
 * @returns The state object with updated button styles
 */
export declare const useTabButtonStyles_unstable: (state: TabState, slot: TabState["root"]) => TabState;

/**
 * Applies styles to the Tab content slot based on its current state.
 *
 * This hook is typically used internally by `useTabStyles_unstable`. You should
 * only use it directly if you're creating a custom `Tab` component.
 *
 * @param state - The Tab component's current state
 * @returns The state object with updated content styles
 */
export declare const useTabContentStyles_unstable: (state: TabState) => TabState;

/**
 * Applies styles for the Tab indicator based on its current state.
 *
 * This hook is typically used internally by `useTabStyles_unstable`. You should
 * only use it directly if you're creating a custom `Tab` component.
 *
 * @param state - The `Tab` component's current state
 * @returns The state object with updated button styles
 */
export declare const useTabIndicatorStyles_unstable: (state: TabState) => TabState;

/**
 * Create the state required to render TabList.
 *
 * The returned state can be modified with hooks such as useTabListStyles_unstable,
 * before being passed to renderTabList_unstable.
 *
 * @param props - props from this instance of TabList
 * @param ref - reference to root HTMLElement of TabList
 */
export declare const useTabList_unstable: (props: TabListProps, ref: React_2.Ref<HTMLElement>) => TabListState;

export declare const useTabListContext_unstable: <T>(selector: ContextSelector<TabListContextValue, T>) => T;

export declare function useTabListContextValues_unstable(state: TabListState): TabListContextValues;

/**
 * Apply styling to the TabList slots based on the state
 */
export declare const useTabListStyles_unstable: (state: TabListState) => TabListState;

/**
 * Apply styling to the Tab slots based on the state
 */
export declare const useTabStyles_unstable: (state: TabState) => TabState;

export { }
