'use client';
import * as React from 'react';
import { ArrowLeft, ArrowRight, Enter, Escape, Shift, Space } from '@fluentui/keyboard-keys';
import { useEventCallback } from '@fluentui/react-utilities';
import { useFocusFinders, useTabsterAttributes } from '@fluentui/react-tabster';
const STEP = 20;
const PRECISION_MODIFIER = Shift;
const PRECISION_FACTOR = 1 / 4;
export function useKeyboardResizing(columnResizeState) {
    const [columnId, setColumnId] = React.useState();
    const onChangeRef = React.useRef(undefined);
    const { findPrevFocusable } = useFocusFinders();
    const columnResizeStateRef = React.useRef(columnResizeState);
    React.useEffect(()=>{
        columnResizeStateRef.current = columnResizeState;
    }, [
        columnResizeState
    ]);
    const [resizeHandleRefs] = React.useState(()=>new Map());
    const keyboardHandler = useEventCallback((event)=>{
        if (!columnId) {
            return;
        }
        const width = columnResizeStateRef.current.getColumnWidth(columnId);
        const precisionModifier = event.getModifierState(PRECISION_MODIFIER);
        const stopEvent = ()=>{
            event.preventDefault();
            event.stopPropagation();
        };
        switch(event.key){
            case ArrowLeft:
                stopEvent();
                columnResizeStateRef.current.setColumnWidth(event.nativeEvent, {
                    columnId,
                    width: width - (precisionModifier ? STEP * PRECISION_FACTOR : STEP)
                });
                return;
            case ArrowRight:
                stopEvent();
                columnResizeStateRef.current.setColumnWidth(event.nativeEvent, {
                    columnId,
                    width: width + (precisionModifier ? STEP * PRECISION_FACTOR : STEP)
                });
                return;
            case Space:
            case Enter:
            case Escape:
                var // Just blur here, the onBlur handler will take care of the rest (disableInteractiveMode).
                _resizeHandleRefs_get_current, _resizeHandleRefs_get;
                stopEvent();
                (_resizeHandleRefs_get = resizeHandleRefs.get(columnId)) === null || _resizeHandleRefs_get === void 0 ? void 0 : (_resizeHandleRefs_get_current = _resizeHandleRefs_get.current) === null || _resizeHandleRefs_get_current === void 0 ? void 0 : _resizeHandleRefs_get_current.blur();
                break;
        }
    });
    const enableInteractiveMode = React.useCallback((colId)=>{
        var _onChangeRef_current, _resizeHandleRefs_get;
        setColumnId(colId);
        (_onChangeRef_current = onChangeRef.current) === null || _onChangeRef_current === void 0 ? void 0 : _onChangeRef_current.call(onChangeRef, colId, true);
        const handle = (_resizeHandleRefs_get = resizeHandleRefs.get(colId)) === null || _resizeHandleRefs_get === void 0 ? void 0 : _resizeHandleRefs_get.current;
        if (handle) {
            handle.setAttribute('tabindex', '-1');
            handle.tabIndex = -1;
            handle.focus();
        }
    }, [
        resizeHandleRefs
    ]);
    const disableInteractiveMode = React.useCallback(()=>{
        var // Notify the onChange listener that we are disabling interactive mode.
        _onChangeRef_current, _resizeHandleRefs_get;
        if (!columnId) {
            return;
        }
        (_onChangeRef_current = onChangeRef.current) === null || _onChangeRef_current === void 0 ? void 0 : _onChangeRef_current.call(onChangeRef, columnId, false);
        // Find the previous focusable element (table header button) and focus it.
        const el = (_resizeHandleRefs_get = resizeHandleRefs.get(columnId)) === null || _resizeHandleRefs_get === void 0 ? void 0 : _resizeHandleRefs_get.current;
        if (el) {
            var _findPrevFocusable;
            (_findPrevFocusable = findPrevFocusable(el)) === null || _findPrevFocusable === void 0 ? void 0 : _findPrevFocusable.focus(); // Focus the previous focusable element (header button).
            el.removeAttribute('tabindex');
        }
        setColumnId(undefined);
    }, [
        columnId,
        findPrevFocusable,
        resizeHandleRefs
    ]);
    const toggleInteractiveMode = (colId, onChange)=>{
        onChangeRef.current = onChange;
        if (!columnId) {
            enableInteractiveMode(colId);
        } else if (colId && columnId !== colId) {
            enableInteractiveMode(colId);
            setColumnId(colId);
        } else {
            disableInteractiveMode();
        }
    };
    const getKeyboardResizingRef = React.useCallback((colId)=>{
        const ref = resizeHandleRefs.get(colId) || React.createRef();
        resizeHandleRefs.set(colId, ref);
        return ref;
    }, [
        resizeHandleRefs
    ]);
    // This makes sure the left and right arrow keys are ignored in tabster,
    // so that they can be used for resizing.
    const tabsterAttrs = useTabsterAttributes({
        focusable: {
            ignoreKeydown: {
                ArrowLeft: true,
                ArrowRight: true
            }
        }
    });
    return {
        toggleInteractiveMode,
        columnId,
        getKeyboardResizingProps: React.useCallback((colId, currentWidth)=>({
                onKeyDown: keyboardHandler,
                onBlur: disableInteractiveMode,
                ref: getKeyboardResizingRef(colId),
                role: 'separator',
                'aria-label': 'Resize column',
                'aria-valuetext': `${currentWidth} pixels`,
                'aria-hidden': colId === columnId ? false : true,
                tabIndex: colId === columnId ? 0 : undefined,
                ...tabsterAttrs
            }), [
            columnId,
            disableInteractiveMode,
            getKeyboardResizingRef,
            keyboardHandler,
            tabsterAttrs
        ])
    };
}
