import type { GriffelStyle } from '@griffel/react';
import type { JSXElement } from '@fluentui/react-utilities';
import * as React_2 from 'react';
import type { RefObjectFunction } from '@fluentui/react-utilities';

export declare type Alignment = 'top' | 'bottom' | 'start' | 'end' | 'center';

export declare type AutoSize = 'height' | 'height-always' | 'width' | 'width-always' | 'always' | boolean;

/**
 * @deprecated use PositioningBoundary instead
 */
export declare type Boundary = PositioningBoundary;

/**
 * @internal
 * Creates CSS styles to size the arrow created by createArrowStyles to the given height.
 *
 * Use this when you need to create classes for several different arrow sizes. If you only need a
 * constant arrow size, you can pass the `arrowHeight` param to createArrowStyles instead.
 */
export declare function createArrowHeightStyles(arrowHeight: number): GriffelStyle;

/**
 * @internal
 * Helper that creates a makeStyles rule for an arrow element.
 * For runtime arrow size toggling simply create extra classnames to apply to the arrow element
 *
 * ```ts
 *   makeStyles({
 *     arrowWithSize: createArrowStyles({ arrowHeight: 6 }),
 *
 *     arrowWithoutSize: createArrowStyles({ arrowHeight: undefined }),
 *     mediumArrow: createArrowHeightStyles(4),
 *     smallArrow: createArrowHeightStyles(2),
 *   })
 *   ...
 *
 *   state.arrowWithSize.className = styles.arrowWithSize;
 *   state.arrowWithoutSize.className = mergeClasses(
 *     styles.arrowWithoutSize,
 *     state.smallArrow && styles.smallArrow,
 *     state.mediumArrow && styles.mediumArrow,
 *   )
 * ```
 */
export declare function createArrowStyles(options: CreateArrowStylesOptions): GriffelStyle;

/**
 * @internal
 * Options parameter for the createArrowStyles function
 */
export declare type CreateArrowStylesOptions = {
    /**
     * The height of the arrow from the base to the tip, in px. The base width of the arrow is always twice its height.
     *
     * This can be undefined to leave out the arrow size styles. You must then add styles created by
     * createArrowHeightStyles to set the arrow's size correctly. This can be useful if the arrow can be different sizes.
     */
    arrowHeight: number | undefined;
    /**
     * The borderWidth of the arrow. Should be the same borderWidth as the parent element.
     *
     * @defaultvalue 1px
     */
    borderWidth?: GriffelStyle['borderBottomWidth'];
    /**
     * The borderStyle for the arrow. Should be the same borderStyle as the parent element.
     *
     * @defaultvalue solid
     */
    borderStyle?: GriffelStyle['borderBottomStyle'];
    /**
     * The borderColor of the arrow. Should be the same borderColor as the parent element.
     *
     * @defaultvalue tokens.colorTransparentStroke
     */
    borderColor?: GriffelStyle['borderBottomColor'];
};

/**
 * Creates animation styles so that positioned elements slide in from the main axis
 * @param mainAxis - distance than the element sides for
 * @returns Griffel styles to spread to a slot
 */
export declare function createSlideStyles(mainAxis: number): GriffelStyle;

/**
 * Creates a virtual element based on the position of a click event
 * Can be used as a target for popper in scenarios such as context menus
 */
export declare function createVirtualElementFromClick(nativeEvent: MouseEvent): PositioningVirtualElement;

/**
 * Generally when adding an arrow to popper, it's necessary to offset the position of the popper by the
 * height of the arrow. A simple utility to merge a provided offset with an arrow height to return the final offset
 *
 * @internal
 * @param userOffset - The offset provided by the user
 * @param arrowHeight - The height of the arrow in px
 * @returns User offset augmented with arrow height
 */
export declare function mergeArrowOffset(userOffset: Offset | undefined | null, arrowHeight: number): Offset;

export declare type Offset = OffsetFunction | OffsetObject | OffsetShorthand;

export declare type OffsetFunction = (param: OffsetFunctionParam) => OffsetObject | OffsetShorthand;

export declare type OffsetFunctionParam = {
    positionedRect: PositioningRect;
    targetRect: PositioningRect;
    position: Position;
    alignment?: Alignment;
};

export declare type OffsetObject = {
    crossAxis?: number;
    mainAxis: number;
};

export declare type OffsetShorthand = number;

export declare type Position = 'above' | 'below' | 'before' | 'after';

export declare type PositioningBoundary = PositioningRect | HTMLElement | Array<HTMLElement> | 'clippingParents' | 'scrollParent' | 'window';

export declare type PositioningConfigurationFn = (params: {
    container: HTMLElement;
    arrow: HTMLElement | null;
    options: PositioningConfigurationFnOptions;
}) => PositioningConfigurationFnOptions;

export declare type PositioningConfigurationFnOptions = Omit<PositioningOptions, 'enabled' | 'onPositioningEnd' | 'positionFixed'>;

/**
 * A context provider for the positioning configuration.
 *
 * Accepts a function that takes the positioning options and returns them modified.
 */
export declare const PositioningConfigurationProvider: React_2.Provider<PositioningConfigurationFn | undefined>;

export declare type PositioningImperativeRef = {
    /**
     * Updates the position imperatively.
     * Useful when the position of the target changes from other factors than scrolling of window resize.
     */
    updatePosition: () => void;
    /**
     * Sets the target and updates positioning imperatively.
     * Useful for avoiding double renders with the target option.
     */
    setTarget: (target: TargetElement | null) => void;
};

/**
 * Internal options for positioning
 */
declare interface PositioningOptions {
    /** Alignment for the component. Only has an effect if used with the @see position option */
    align?: Alignment;
    /** The element which will define the boundaries of the positioned element for the flip behavior. */
    flipBoundary?: PositioningBoundary | null;
    /** The element which will define the boundaries of the positioned element for the overflow behavior. */
    overflowBoundary?: PositioningBoundary | null;
    /**
     * Applies a padding to the overflow bounadry, so that overflow is detected earlier before the
     * positioned surface hits the overflow boundary.
     */
    overflowBoundaryPadding?: number | Partial<{
        top: number;
        end: number;
        bottom: number;
        start: number;
    }>;
    /**
     * Position for the component. Position has higher priority than align. If position is vertical ('above' | 'below')
     * and align is also vertical ('top' | 'bottom') or if both position and align are horizontal ('before' | 'after'
     * and 'start' | 'end' respectively),
     * then provided value for 'align' will be ignored and 'center' will be used instead.
     */
    position?: Position;
    /**
     * Enables the position element to be positioned with 'fixed' (default value is position: 'absolute')
     * @default false
     * @deprecated use `strategy` instead
     */
    positionFixed?: boolean;
    /**
     * Specifies the type of CSS position property to use.
     * @default absolute
     */
    strategy?: 'absolute' | 'fixed';
    /**
     * Lets you displace a positioned element from its reference element.
     * This can be useful if you need to apply some margin between them or if you need to fine tune the
     * position according to some custom logic.
     */
    offset?: Offset;
    /**
     * Defines padding between the corner of the popup element and the arrow.
     * Use to prevent the arrow from overlapping a rounded corner, for example.
     */
    arrowPadding?: number;
    /**
     * Applies styles on the positioned element to fit it within the available space in viewport.
     * - true: set styles for max height/width.
     * - 'height': set styles for max height.
     * - 'width'': set styles for max width.
     * Note that options 'always'/'height-always'/'width-always' are now obsolete, and equivalent to true/'height'/'width'.
     */
    autoSize?: AutoSize;
    /**
     * Modifies position and alignment to cover the target
     */
    coverTarget?: boolean;
    /**
     * Disables automatic repositioning of the component; it will always be placed according to the values of `align` and
     * `position` props, regardless of the size of the component, the reference element or the viewport.
     */
    pinned?: boolean;
    /**
     * When the reference element or the viewport is outside viewport allows a positioned element to be fully in viewport.
     * "all" enables this behavior for all axis.
     */
    unstable_disableTether?: boolean | 'all';
    /**
     * If flip fails to stop the positioned element from overflowing
     * its boundaries, use a specified fallback positions.
     */
    fallbackPositions?: PositioningShorthandValue[];
    /**
     * Modifies whether popover is positioned using transform.
     * @default true
     */
    useTransform?: boolean;
    /**
     * If false, does not position anything
     */
    enabled?: boolean;
    /**
     * When set, the positioned element matches the chosen dimension(s) of the target element
     */
    matchTargetSize?: 'width';
    /**
     * Called when a position update has finished. Multiple position updates can happen in a single render,
     * since positioning happens outside of the React lifecycle.
     *
     * It's also possible to listen to the custom DOM event `fui-positioningend`
     */
    onPositioningEnd?: () => void;
    /**
     * Disables the resize observer that updates position on target or dimension change
     */
    disableUpdateOnResize?: boolean;
    /**
     * When true, the positioned element will shift to cover the target element when there's not enough space.
     * @default false
     */
    shiftToCoverTarget?: boolean;
}

/**
 * Public api that allows components using react-positioning to specify positioning options
 */
export declare interface PositioningProps extends Pick<PositioningOptions, 'align' | 'arrowPadding' | 'autoSize' | 'coverTarget' | 'fallbackPositions' | 'flipBoundary' | 'offset' | 'overflowBoundary' | 'overflowBoundaryPadding' | 'pinned' | 'position' | 'strategy' | 'useTransform' | 'matchTargetSize' | 'onPositioningEnd' | 'disableUpdateOnResize' | 'shiftToCoverTarget'> {
    /** An imperative handle to Popper methods. */
    positioningRef?: React_2.Ref<PositioningImperativeRef>;
    /**
     * Manual override for the target element. Useful for scenarios where a component accepts user prop to override target
     */
    target?: TargetElement | null;
}

export declare type PositioningRect = {
    width: number;
    height: number;
    x: number;
    y: number;
};

export declare type PositioningShorthand = PositioningProps | PositioningShorthandValue;

export declare type PositioningShorthandValue = 'above' | 'above-start' | 'above-end' | 'below' | 'below-start' | 'below-end' | 'before' | 'before-top' | 'before-bottom' | 'after' | 'after-top' | 'after-bottom';

export declare type PositioningVirtualElement = {
    getBoundingClientRect: () => {
        x: number;
        y: number;
        top: number;
        left: number;
        bottom: number;
        right: number;
        width: number;
        height: number;
    };
    contextElement?: Element;
};

export declare function resolvePositioningShorthand(shorthand: PositioningShorthand | undefined | null): Readonly<PositioningProps>;

export declare type SetVirtualMouseTarget = (event: React_2.MouseEvent | MouseEvent | undefined | null) => void;

declare type TargetElement = HTMLElement | PositioningVirtualElement;

/**
 * @internal
 */
export declare function usePositioning(options: PositioningProps & PositioningOptions): UsePositioningReturn;

/**
 * @internal
 * A state hook that manages a popper virtual element from mouseevents.
 * Useful for scenarios where a component needs to be positioned by mouse click (e.g. contextmenu)
 * React synthetic events are not persisted by this hook
 *
 * @param initialState - initializes a user provided state similare to useState
 * @returns state and dispatcher for a Popper virtual element that uses native/synthetic mouse events
 */
export declare const usePositioningMouseTarget: (initialState?: PositioningVirtualElement | (() => PositioningVirtualElement)) => readonly [PositioningVirtualElement | undefined, SetVirtualMouseTarget];

declare interface UsePositioningReturn {
    targetRef: React_2.MutableRefObject<any>;
    containerRef: React_2.MutableRefObject<any>;
    arrowRef: React_2.MutableRefObject<any>;
}

export declare function useSafeZoneArea({ debug, disabled, onSafeZoneEnter, onSafeZoneMove, onSafeZoneLeave, onSafeZoneTimeout, timeout, }?: UseSafeZoneOptions): {
    containerRef: RefObjectFunction<HTMLElement>;
    targetRef: RefObjectFunction<HTMLElement>;
    elementToRender: JSXElement | null;
};

export declare type UseSafeZoneOptions = {
    /** Enables debug mode: makes drawn shapes visible. */
    debug?: boolean;
    /** Disables the safe zone area. */
    disabled?: boolean;
    /** The time in milliseconds to wait before clearing the safe zone. */
    timeout?: number;
    /** Called when the mouse enters the safe zone. */
    onSafeZoneEnter?: (e: React_2.MouseEvent) => void;
    /** Called when the mouse moves within the safe zone. */
    onSafeZoneMove?: (e: React_2.MouseEvent) => void;
    /** Called when the mouse leaves the safe zone. */
    onSafeZoneLeave?: (e: React_2.MouseEvent) => void;
    /** Called when the safe zone times out, even if a cursor is still over a safe zone. */
    onSafeZoneTimeout?: () => void;
};

export { }
