'use client';
"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "usePortalMountNode", {
    enumerable: true,
    get: function() {
        return usePortalMountNode;
    }
});
const _interop_require_wildcard = require("@swc/helpers/_/_interop_require_wildcard");
const _react = /*#__PURE__*/ _interop_require_wildcard._(require("react"));
const _reactsharedcontexts = require("@fluentui/react-shared-contexts");
const _react1 = require("@griffel/react");
const _reacttabster = require("@fluentui/react-tabster");
const _usePortalMountNodeStylesstyles = require("./usePortalMountNodeStyles.styles");
const useInsertionEffect = _react['useInsertion' + 'Effect'];
/**
 * Legacy element factory for React 17 and below. It's not safe for concurrent rendering.
 *
 * Creates a new element on a "document.body" to mount portals.
 */ const useLegacyElementFactory = (options)=>{
    'use no memo';
    const { className, dir, focusVisibleRef, targetNode } = options;
    const targetElement = _react.useMemo(()=>{
        if (targetNode === undefined || options.disabled) {
            return null;
        }
        const element = targetNode.ownerDocument.createElement('div');
        targetNode.appendChild(element);
        return element;
    }, [
        targetNode,
        options.disabled
    ]);
    // Heads up!
    // This useMemo() call is intentional for React 17 & below.
    //
    // We don't want to re-create the portal element when its attributes change. This also cannot not be done in an effect
    // because, changing the value of CSS variables after an initial mount will trigger interesting CSS side effects like
    // transitions.
    _react.useMemo(()=>{
        if (!targetElement) {
            return;
        }
        targetElement.className = className;
        targetElement.setAttribute('dir', dir);
        targetElement.setAttribute('data-portal-node', 'true');
        focusVisibleRef.current = targetElement;
    }, [
        className,
        dir,
        targetElement,
        focusVisibleRef
    ]);
    _react.useEffect(()=>{
        return ()=>{
            targetElement === null || targetElement === void 0 ? void 0 : targetElement.remove();
        };
    }, [
        targetElement
    ]);
    return targetElement;
};
const initializeElementFactory = ()=>{
    let currentElement = undefined;
    function get(targetRoot, forceCreation) {
        if (currentElement) {
            return currentElement;
        }
        if (forceCreation) {
            currentElement = targetRoot.ownerDocument.createElement('div');
            targetRoot.appendChild(currentElement);
        }
        return currentElement;
    }
    function dispose() {
        if (currentElement) {
            currentElement.remove();
            currentElement = undefined;
        }
    }
    return {
        get,
        dispose
    };
};
/**
 * This is a modern element factory for React 18 and above. It is safe for concurrent rendering.
 *
 * It abuses the fact that React will mount DOM once (unlike hooks), so by using a proxy we can intercept:
 * - the `remove()` method (we call it in `useEffect()`) and remove the element only when the portal is unmounted
 * - all other methods (and properties) will be called by React once a portal is mounted
 */ const useModernElementFactory = (options)=>{
    'use no memo';
    const { className, dir, focusVisibleRef, targetNode } = options;
    const [elementFactory] = _react.useState(initializeElementFactory);
    const elementProxy = _react.useMemo(()=>{
        if (targetNode === undefined || options.disabled) {
            return null;
        }
        return new Proxy({}, {
            get (_, property) {
                // Heads up!
                // `createPortal()` performs a check for `nodeType` property to determine if the mount node is a valid DOM node
                // before mounting the portal. We hardcode the value to `Node.ELEMENT_NODE` to pass this check and avoid
                // premature node creation
                if (property === 'nodeType') {
                    // Can't use the `Node.ELEMENT_NODE` as it's a browser API and  not available in all environments, e.g SSR
                    return 1; // `Node.ELEMENT_NODE`
                }
                // Heads up!
                // We intercept the `remove()` method to remove the mount node only when portal has been unmounted already.
                if (property === 'remove') {
                    const targetElement = elementFactory.get(targetNode, false);
                    if (targetElement) {
                        // If the mountElement has children, the portal is still mounted, otherwise we can dispose of it
                        const portalHasNoChildren = targetElement.childNodes.length === 0;
                        if (portalHasNoChildren) {
                            elementFactory.dispose();
                        }
                    }
                    return ()=>{
                    // Always return a no-op function to avoid errors in the code
                    };
                }
                const targetElement = elementFactory.get(targetNode, true);
                const targetProperty = targetElement ? targetElement[property] : undefined;
                if (typeof targetProperty === 'function') {
                    return targetProperty.bind(targetElement);
                }
                return targetProperty;
            },
            set (_, property, value) {
                const ignoredProperty = property === '_virtual' || property === 'focusVisible';
                // We should use the `elementFactory.get(targetNode, !ignoredProperty)`,
                // but TypeScript requires a literal `true` or `false` for the overload signature.
                // This workaround ensures the correct overload is called and avoids TypeScript errors.
                const targetElement = ignoredProperty ? elementFactory.get(targetNode, false) : elementFactory.get(targetNode, true);
                if (ignoredProperty && !targetElement) {
                    // We ignore the `_virtual` and `focusVisible` properties to avoid conflicts with the proxy
                    return true;
                }
                if (targetElement) {
                    Object.assign(targetElement, {
                        [property]: value
                    });
                    return true;
                }
                return false;
            }
        });
    }, [
        elementFactory,
        targetNode,
        options.disabled
    ]);
    useInsertionEffect(()=>{
        if (!elementProxy) {
            return;
        }
        const classesToApply = className.split(' ').filter(Boolean);
        elementProxy.classList.add(...classesToApply);
        elementProxy.setAttribute('dir', dir);
        elementProxy.setAttribute('data-portal-node', 'true');
        focusVisibleRef.current = elementProxy;
        return ()=>{
            elementProxy.classList.remove(...classesToApply);
            elementProxy.removeAttribute('dir');
        };
    }, [
        className,
        dir,
        elementProxy,
        focusVisibleRef
    ]);
    _react.useEffect(()=>{
        return ()=>{
            elementProxy === null || elementProxy === void 0 ? void 0 : elementProxy.remove();
        };
    }, [
        elementProxy
    ]);
    return elementProxy;
};
/**
 * Element factory based on the React version.
 *
 * React 17 and below:
 * - useLegacyElementFactory
 *
 * React 18 and above:
 * - useModernElementFactory
 */ const useElementFactory = useInsertionEffect ? useModernElementFactory : useLegacyElementFactory;
const usePortalMountNode = (options)=>{
    'use no memo';
    const { targetDocument, dir } = (0, _reactsharedcontexts.useFluent_unstable)();
    const mountNode = (0, _reactsharedcontexts.usePortalMountNode)();
    // eslint-disable-next-line @typescript-eslint/no-deprecated
    const focusVisibleRef = (0, _reacttabster.useFocusVisible)();
    const classes = (0, _usePortalMountNodeStylesstyles.usePortalMountNodeStylesStyles)();
    const themeClassName = (0, _reactsharedcontexts.useThemeClassName_unstable)();
    const factoryOptions = {
        dir,
        disabled: options.disabled,
        focusVisibleRef,
        className: (0, _react1.mergeClasses)(themeClassName, classes.root, options.className),
        targetNode: mountNode !== null && mountNode !== void 0 ? mountNode : targetDocument === null || targetDocument === void 0 ? void 0 : targetDocument.body
    };
    return useElementFactory(factoryOptions);
};
