import { Avatar } from '@fluentui/react-avatar';
import type { ComponentProps } from '@fluentui/react-utilities';
import type { ComponentState } from '@fluentui/react-utilities';
import type { ForwardRefComponent } from '@fluentui/react-utilities';
import type { JSXElement } from '@fluentui/react-utilities';
import { PresenceBadge } from '@fluentui/react-badge';
import * as React_2 from 'react';
import type { Slot } from '@fluentui/react-utilities';
import type { SlotClassNames } from '@fluentui/react-utilities';

/**
 * A Persona is a visual representation of a person or status that showcases an Avatar, PresenceBadge,
 * or an Avatar with a PresenceBadge.
 */
export declare const Persona: ForwardRefComponent<PersonaProps>;

export declare const personaClassNames: SlotClassNames<PersonaSlots>;

/**
 * Persona Props
 */
export declare type PersonaProps = ComponentProps<PersonaSlots> & {
    /**
     * The name of the person or entity represented by the Persona.
     *
     * When `primaryText` is not provided, this will be used as the default value for `primaryText`.
     */
    name?: string;
    /**
     * Whether to display only the presence.
     *
     * @default false
     */
    presenceOnly?: boolean;
    /**
     * The size of a Persona and its text.
     *
     * @default medium
     */
    size?: 'extra-small' | 'small' | 'medium' | 'large' | 'extra-large' | 'huge';
    /**
     * The position of the text relative to the avatar/presence.
     *
     * @default after
     */
    textPosition?: 'after' | 'before' | 'below';
    /**
     * The vertical alignment of the text relative to the avatar/presence.
     *
     * @default start
     */
    textAlignment?: 'center' | 'start';
};

export declare type PersonaSlots = {
    root: NonNullable<Slot<'div'>>;
    /**
     * Avatar to display.
     *
     * If a PresenceBadge and an Avatar are provided, the Avatar will display the PresenceBadge as its presence.
     */
    avatar?: Slot<typeof Avatar>;
    /**
     * PresenceBadge to display.
     *
     * If `presenceOnly` is true, the PresenceBadge will be displayed instead of the Avatar.
     */
    presence?: Slot<typeof PresenceBadge>;
    /**
     * The first line of text in the Persona, larger than the rest of the lines.
     *
     * `primaryText` defaults to the `name` prop. We recomend to only use `name`, use `primaryText` when the text is
     *  different than the `name` prop.
     */
    primaryText?: Slot<'span'>;
    /**
     * The second line of text in the Persona.
     */
    secondaryText?: Slot<'span'>;
    /**
     * The third line of text in the Persona.
     */
    tertiaryText?: Slot<'span'>;
    /**
     * The fourth line of text in the Persona.
     */
    quaternaryText?: Slot<'span'>;
};

/**
 * State used in rendering Persona
 */
export declare type PersonaState = ComponentState<PersonaSlots> & Required<Pick<PersonaProps, 'presenceOnly' | 'size' | 'textAlignment' | 'textPosition'>> & {
    /**
     * The number of text lines used.
     */
    numTextLines: number;
};

/**
 * Render the final JSX of Persona
 */
export declare const renderPersona_unstable: (state: PersonaState) => JSXElement;

/**
 * Create the state required to render Persona.
 *
 * The returned state can be modified with hooks such as usePersonaStyles_unstable,
 * before being passed to renderPersona_unstable.
 *
 * @param props - props from this instance of Persona
 * @param ref - reference to root HTMLElement of Persona
 */
export declare const usePersona_unstable: (props: PersonaProps, ref: React_2.Ref<HTMLElement>) => PersonaState;

/**
 * Apply styling to the Persona slots based on the state
 */
export declare const usePersonaStyles_unstable: (state: PersonaState) => PersonaState;

export { }
