import { ContextSelector } from '@fluentui/react-context-selector';
import type { ObserveOptions } from '@fluentui/priority-overflow';
import type { OnUpdateOverflow } from '@fluentui/priority-overflow';
import type { OverflowDividerEntry } from '@fluentui/priority-overflow';
import { OverflowGroupState } from '@fluentui/priority-overflow';
import type { OverflowItemEntry } from '@fluentui/priority-overflow';
import * as React_2 from 'react';

export declare const DATA_OVERFLOW_DIVIDER = "data-overflow-divider";

export declare const DATA_OVERFLOW_ITEM = "data-overflow-item";

export declare const DATA_OVERFLOW_MENU = "data-overflow-menu";

export declare const DATA_OVERFLOWING = "data-overflowing";

export declare interface OnOverflowChangeData extends OverflowState {
}

/**
 * Provides an OverflowContext for OverflowItem descendants.
 */
export declare const Overflow: React_2.ForwardRefExoticComponent<Partial<Pick<ObserveOptions, "padding" | "overflowDirection" | "overflowAxis" | "minimumVisible" | "hasHiddenItems">> & {
    children: React_2.ReactElement;
    onOverflowChange?: (ev: null, data: OverflowState) => void;
} & React_2.RefAttributes<unknown>>;

/**
 * @internal
 */
declare interface OverflowContextValue {
    itemVisibility: Record<string, boolean>;
    groupVisibility: Record<string, OverflowGroupState>;
    hasOverflow: boolean;
    registerItem: (item: OverflowItemEntry) => () => void;
    registerOverflowMenu: (el: HTMLElement) => () => void;
    registerDivider: (divider: OverflowDividerEntry) => () => void;
    updateOverflow: (padding?: number) => void;
}

/**
 * Attaches overflow item behavior to its child registered with the OverflowContext.
 * It does not render an element of its own.
 */
export declare const OverflowDivider: React_2.ForwardRefExoticComponent<OverflowDividerProps & React_2.RefAttributes<unknown>>;

/**
 * OverflowDividerProps
 */
declare type OverflowDividerProps = {
    /**
     * Assigns the item to a group, group visibility can be watched.
     */
    groupId: string;
    /**
     * The single child that has overflow item behavior attached.
     */
    children: React_2.ReactElement;
};

/**
 * Attaches overflow item behavior to its child registered with the OverflowContext.
 * It does not render an element of its own.
 *
 * Behaves similarly to other `*Trigger` components in Fluent UI React.
 */
export declare const OverflowItem: React_2.ForwardRefExoticComponent<OverflowItemProps & React_2.RefAttributes<unknown>>;

/**
 * OverflowItemProps
 */
export declare type OverflowItemProps = {
    /**
     * The unique identifier for the item used by the overflow manager.
     */
    id: string;
    /**
     * Assigns the item to a group, group visibility can be watched.
     */
    groupId?: string;
    /**
     * A higher priority means the item overflows later.
     */
    priority?: number;
    /**
     * The single child that has overflow item behavior attached.
     */
    children: React_2.ReactElement;
};

/**
 * Overflow Props
 */
export declare type OverflowProps = Partial<Pick<ObserveOptions, 'overflowAxis' | 'overflowDirection' | 'padding' | 'minimumVisible' | 'hasHiddenItems'>> & {
    children: React_2.ReactElement;
    onOverflowChange?: (ev: null, data: OverflowState) => void;
};

declare interface OverflowState {
    hasOverflow: boolean;
    itemVisibility: Record<string, boolean>;
    groupVisibility: Record<string, OverflowGroupState>;
}

/**
 * @param id - unique identifier for a group of overflow items
 * @returns visibility state of the group
 */
export declare function useIsOverflowGroupVisible(id: string): OverflowGroupState;

/**
 * @param id - unique identifier for the item used by the overflow manager
 * @returns visibility state of an overflow item
 */
export declare function useIsOverflowItemVisible(id: string): boolean;

/**
 * @internal
 * @param update - Callback when overflow state changes
 * @param options - Options to configure the overflow container
 * @returns - ref to attach to an intrinsic HTML element and imperative functions
 */
export declare const useOverflowContainer: <TElement extends HTMLElement>(update: OnUpdateOverflow, options: Omit<ObserveOptions, "onUpdateOverflow">) => UseOverflowContainerReturn<TElement>;

/**
 * @internal
 */
export declare interface UseOverflowContainerReturn<TElement extends HTMLElement> extends Pick<OverflowContextValue, 'registerItem' | 'updateOverflow' | 'registerOverflowMenu' | 'registerDivider'> {
    /**
     * Ref to apply to the container that will overflow
     */
    containerRef: React_2.RefObject<TElement | null>;
}

/**
 * @internal
 */
export declare const useOverflowContext: <SelectedValue>(selector: ContextSelector<OverflowContextValue, SelectedValue>) => SelectedValue;

/**
 * @returns Number of items that are overflowing
 */
export declare const useOverflowCount: () => number;

/**
 * @internal
 * Registers an overflow item
 * @param groupId - assigns the item to a group, group visibility can be watched
 * @returns ref to assign to an intrinsic HTML element
 */
export declare function useOverflowDivider<TElement extends HTMLElement>(groupId?: string): React_2.RefObject<TElement | null>;

/**
 * @internal
 * Registers an overflow item
 * @param id - unique identifier for the item used by the overflow manager
 * @param priority - higher priority means the item overflows later
 * @param groupId - assigns the item to a group, group visibility can be watched
 * @returns ref to assign to an intrinsic HTML element
 */
export declare function useOverflowItem<TElement extends HTMLElement>(id: string, priority?: number, groupId?: string): React_2.RefObject<TElement | null>;

export declare function useOverflowMenu<TElement extends HTMLElement>(id?: string): {
    ref: React_2.MutableRefObject<TElement | null>;
    overflowCount: number;
    isOverflowing: boolean;
};

/**
 * A hook that returns the visibility status of all items and groups.
 *
 * ⚠️ Heads up!
 *
 * This hook will cause the component it is in to re-render for every single time an item overflows or becomes
 * visible - use with caution
 * @returns visibility status of all items and groups
 */
export declare function useOverflowVisibility(): {
    itemVisibility: Record<string, boolean>;
    groupVisibility: Record<string, OverflowGroupState>;
};

export { }
