import type { JSXElement } from '@fluentui/react-utilities';
import type { JSXIntrinsicElementKeys } from '@fluentui/react-utilities';
import * as React_2 from 'react';
import type { SlotComponentType } from '@fluentui/react-utilities';
import type { SlotRenderFunction } from '@fluentui/react-utilities';

declare type AtomCore = {
    keyframes: Keyframe[];
} & KeyframeEffectOptions;

export declare type AtomMotion = AtomCore & {
    /**
     * Allows to specify a reduced motion version of the animation. If provided, the settings will be used when the
     * user has enabled the reduced motion setting in the operating system (i.e `prefers-reduced-motion` media query is
     * active). If not provided, the duration of the animation will be overridden to be 1ms.
     *
     * Note, if `keyframes` are provided, they will be used instead of the regular `keyframes`.
     */
    reducedMotion?: Partial<AtomCore>;
};

export declare type AtomMotionFn<MotionParams extends Record<string, MotionParam> = {}> = (params: {
    element: HTMLElement;
} & MotionParams) => AtomMotion | AtomMotion[];

/**
 * Creates a component that will animate the children using the provided motion.
 *
 * @param value - A motion definition.
 */
export declare function createMotionComponent<MotionParams extends Record<string, MotionParam> = {}>(value: AtomMotion | AtomMotion[] | AtomMotionFn<MotionParams>): MotionComponent<MotionParams>;

/**
 * Create a new motion component based on another motion component,
 * using the provided variant parameters as defaults.
 *
 * @param component - A component created by `createMotionComponent`.
 * @param variantParams - An object containing the variant parameters to be used as defaults.
 * The variant parameters should match the type of the component's motion parameters.
 * @returns A new motion component that uses the provided variant parameters as defaults.
 * The new component can still accept runtime parameters that override the defaults.
 */
export declare function createMotionComponentVariant<MotionParams extends Record<string, MotionParam> = {}>(component: MotionComponent<MotionParams>, variantParams: Partial<MotionParams>): MotionComponent<MotionParams>;

export declare function createPresenceComponent<MotionParams extends Record<string, MotionParam> = {}>(value: PresenceMotion | PresenceMotionFn<MotionParams>): PresenceComponent<MotionParams>;

/**
 * Create a new presence component based on another presence component,
 * using the provided variant parameters as defaults.
 *
 * @param component - A component created by `createPresenceComponent`.
 * @param variantParams - An object containing the variant parameters to be used as defaults.
 * The variant parameters should match the type of the component's motion parameters.
 * @returns A new presence component that uses the provided variant parameters as defaults.
 * The new component can still accept runtime parameters that override the defaults.
 */
export declare function createPresenceComponentVariant<MotionParams extends Record<string, MotionParam> = {}>(component: PresenceComponent<MotionParams>, variantParams: Partial<MotionParams>): PresenceComponent<MotionParams>;

export declare const curves: {
    readonly curveAccelerateMax: "cubic-bezier(0.9,0.1,1,0.2)";
    readonly curveAccelerateMid: "cubic-bezier(1,0,1,1)";
    readonly curveAccelerateMin: "cubic-bezier(0.8,0,0.78,1)";
    readonly curveDecelerateMax: "cubic-bezier(0.1,0.9,0.2,1)";
    readonly curveDecelerateMid: "cubic-bezier(0,0,0,1)";
    readonly curveDecelerateMin: "cubic-bezier(0.33,0,0.1,1)";
    readonly curveEasyEaseMax: "cubic-bezier(0.8,0,0.2,1)";
    readonly curveEasyEase: "cubic-bezier(0.33,0,0.67,1)";
    readonly curveLinear: "cubic-bezier(0,0,1,1)";
};

export declare const durations: {
    readonly durationUltraFast: 50;
    readonly durationFaster: 100;
    readonly durationFast: 150;
    readonly durationNormal: 200;
    readonly durationGentle: 250;
    readonly durationSlow: 300;
    readonly durationSlower: 400;
    readonly durationUltraSlow: 500;
};

/**
 * @internal A private symbol to store the motion definition on the component for variants.
 */
declare const MOTION_DEFINITION: unique symbol;

export declare const MotionBehaviourProvider: React_2.Provider<MotionBehaviourType | undefined>;

/**
 * Specifies the behaviour of child motion component under @see MotionBehaviourProvider.
 */
declare type MotionBehaviourType = 'skip' | 'default';

export declare type MotionComponent<MotionParams extends Record<string, MotionParam> = {}> = React_2.FC<MotionComponentProps & MotionParams> & {
    [MOTION_DEFINITION]: AtomMotionFn<MotionParams>;
};

export declare type MotionComponentProps = {
    children: JSXElement;
    /** Provides imperative controls for the animation. */
    imperativeRef?: React_2.Ref<MotionImperativeRef | undefined>;
    /**
     * Callback that is called when the whole motion finishes.
     *
     * A motion definition can contain multiple animations and therefore multiple "finish" events. The callback is
     * triggered once all animations have finished with "null" instead of an event object to avoid ambiguity.
     */
    onMotionFinish?: (ev: null) => void;
    /**
     * Callback that is called when the whole motion is cancelled.
     *
     * A motion definition can contain multiple animations and therefore multiple "cancel" events. The callback is
     * triggered once all animations have been cancelled with "null" instead of an event object to avoid ambiguity.
     */
    onMotionCancel?: (ev: null) => void;
    /**
     * Callback that is called when the whole motion starts.
     *
     * A motion definition can contain multiple animations and therefore multiple "start" events. The callback is
     * triggered when the first animation is started. There is no official "start" event with the Web Animations API.
     * so the callback is triggered with "null".
     */
    onMotionStart?: (ev: null) => void;
};

export declare type MotionImperativeRef = {
    /** Sets the playback rate of the animation, where 1 is normal speed. */
    setPlaybackRate: (rate: number) => void;
    /** Sets the state of the animation to running or paused. */
    setPlayState: (state: 'running' | 'paused') => void;
};

/**
 * A motion param should be a primitive value that can be serialized to JSON and could be potentially used a plain
 * dependency for React hooks.
 */
export declare type MotionParam = boolean | number | string;

export declare const motionTokens: {
    curveAccelerateMax: "cubic-bezier(0.9,0.1,1,0.2)";
    curveAccelerateMid: "cubic-bezier(1,0,1,1)";
    curveAccelerateMin: "cubic-bezier(0.8,0,0.78,1)";
    curveDecelerateMax: "cubic-bezier(0.1,0.9,0.2,1)";
    curveDecelerateMid: "cubic-bezier(0,0,0,1)";
    curveDecelerateMin: "cubic-bezier(0.33,0,0.1,1)";
    curveEasyEaseMax: "cubic-bezier(0.8,0,0.2,1)";
    curveEasyEase: "cubic-bezier(0.33,0,0.67,1)";
    curveLinear: "cubic-bezier(0,0,1,1)";
    durationUltraFast: 50;
    durationFaster: 100;
    durationFast: 150;
    durationNormal: 200;
    durationGentle: 250;
    durationSlow: 300;
    durationSlower: 400;
    durationUltraSlow: 500;
};

/**
 * @internal A private symbol to store the motion definition on the component for variants.
 */
declare const PRESENCE_MOTION_DEFINITION: unique symbol;

export declare type PresenceComponent<MotionParams extends Record<string, MotionParam> = {}> = React_2.FC<PresenceComponentProps & MotionParams> & {
    (props: PresenceComponentProps & MotionParams): JSXElement | null;
    [PRESENCE_MOTION_DEFINITION]: PresenceMotionFn<MotionParams>;
    In: React_2.FC<MotionComponentProps & MotionParams>;
    Out: React_2.FC<MotionComponentProps & MotionParams>;
};

export declare type PresenceComponentProps = {
    /**
     * By default, the child component won't execute the "enter" motion when it initially mounts, regardless of the value
     * of "visible". If you desire this behavior, ensure both "appear" and "visible" are set to "true".
     */
    appear?: boolean;
    /** A React element that will be cloned and will have motion effects applied to it. */
    children: JSXElement;
    /** Provides imperative controls for the animation. */
    imperativeRef?: React_2.Ref<MotionImperativeRef | undefined>;
    /**
     * Callback that is called when the whole motion finishes.
     *
     * A motion definition can contain multiple animations and therefore multiple "finish" events. The callback is
     * triggered once all animations have finished with "null" instead of an event object to avoid ambiguity.
     */
    onMotionFinish?: (ev: null, data: {
        direction: PresenceDirection;
    }) => void;
    /**
     * Callback that is called when the whole motion is cancelled. When a motion is cancelled it does not
     * emit a finish event but a specific cancel event
     *
     * A motion definition can contain multiple animations and therefore multiple "finish" events. The callback is
     * triggered once all animations have finished with "null" instead of an event object to avoid ambiguity.
     */
    onMotionCancel?: (ev: null, data: {
        direction: PresenceDirection;
    }) => void;
    /**
     * Callback that is called when the whole motion starts.
     *
     * A motion definition can contain multiple animations and therefore multiple "start" events. The callback is
     * triggered when the first animation is started. There is no official "start" event with the Web Animations API.
     * so the callback is triggered with "null".
     */
    onMotionStart?: (ev: null, data: {
        direction: PresenceDirection;
    }) => void;
    /** Defines whether a component is visible; triggers the "enter" or "exit" motions. */
    visible?: boolean;
    /**
     * By default, the child component remains mounted after it reaches the "finished" state. Set "unmountOnExit" if
     * you prefer to unmount the component after it finishes exiting.
     */
    unmountOnExit?: boolean;
};

export declare type PresenceDirection = 'enter' | 'exit';

export declare class PresenceGroup extends React_2.Component<PresenceGroupProps, PresenceGroupState> {
    private mounted;
    static getDerivedStateFromProps(nextProps: PresenceGroupProps, { childMapping: prevChildMapping, firstRender }: PresenceGroupState): PresenceGroupState;
    constructor(props: PresenceGroupProps, context?: unknown);
    private handleExit;
    componentDidMount(): void;
    componentWillUnmount(): void;
    render(): JSXElement;
}

declare type PresenceGroupChild = {
    element: JSXElement;
    appear: boolean;
    visible: boolean;
    unmountOnExit: boolean;
};

export declare type PresenceGroupChildContextValue = {
    appear: boolean;
    visible: boolean;
    unmountOnExit: boolean;
    onExit: () => void;
};

declare type PresenceGroupChildMapping = Record<string, PresenceGroupChild>;

export declare const PresenceGroupChildProvider: React_2.Provider<PresenceGroupChildContextValue | undefined>;

declare type PresenceGroupProps = {
    children: React_2.ReactNode;
};

declare type PresenceGroupState = {
    childMapping: PresenceGroupChildMapping;
    firstRender: boolean;
};

export declare type PresenceMotion = Record<PresenceDirection, AtomMotion | AtomMotion[]>;

export declare type PresenceMotionFn<MotionParams extends Record<string, MotionParam> = {}> = (params: {
    element: HTMLElement;
} & MotionParams) => PresenceMotion;

export declare function presenceMotionSlot<MotionParams extends Record<string, MotionParam> = {}>(motion: PresenceMotionSlotProps<MotionParams> | null | undefined, options: {
    elementType: React_2.FC<PresenceComponentProps & MotionParams>;
    defaultProps: PresenceMotionSlotRenderProps & MotionParams;
}): SlotComponentType<PresenceMotionSlotRenderProps & MotionParams>;

export declare type PresenceMotionSlotProps<MotionParams extends Record<string, MotionParam> = {}> = Pick<PresenceComponentProps, 'imperativeRef' | 'onMotionFinish' | 'onMotionStart'> & {
    /**
     * @deprecated Do not use. Presence Motion Slots do not support intrinsic elements.
     *
     * If you want to override the animation, use the children render function instead.
     */
    as?: JSXIntrinsicElementKeys;
    children?: SlotRenderFunction<PresenceMotionSlotRenderProps & MotionParams & {
        children: JSXElement;
    }>;
};

/**
 * @internal
 */
declare type PresenceMotionSlotRenderProps = Pick<PresenceComponentProps, 'appear' | 'onMotionFinish' | 'onMotionStart' | 'unmountOnExit' | 'visible'>;

export declare const usePresenceGroupChildContext: () => PresenceGroupChildContextValue | undefined;

export { }
