import { PresenceComponent } from '@fluentui/react-motion';

/**
 * Common opacity animation parameter for motion components.
 */
declare type AnimateOpacity = {
    /** Whether to animate the opacity. Defaults to `true`. */
    animateOpacity?: boolean;
};

declare type Axis3D = 'x' | 'y' | 'z';

/**
 * Base presence parameters combining duration, easing, and delay for motion components.
 */
declare type BasePresenceParams = PresenceDuration & PresenceEasing & PresenceDelay;

/** A React component that applies blur in/out transitions to its children. */
export declare const Blur: PresenceComponent<BlurParams>;

export declare type BlurParams = BasePresenceParams & AnimateOpacity & {
    /** The blur radius with units to animate from. Defaults to '10px'. */
    fromRadius?: string;
    /** The blur radius with units to animate to. Defaults to '0px'. */
    toRadius?: string;
};

/** A React component that applies collapse/expand transitions to its children. */
export declare const Collapse: PresenceComponent<CollapseParams>;

/** A React component that applies collapse/expand transitions with delayed fade to its children. */
export declare const CollapseDelayed: PresenceComponent<CollapseParams>;

/**
 * Duration properties for granular timing control in Collapse animations.
 */
export declare type CollapseDurations = {
    /** Time (ms) for the size animation during enter. Defaults to `duration` for unified timing. */
    sizeDuration?: number;
    /** Time (ms) for the opacity animation during enter. Defaults to `sizeDuration` for synchronized timing. */
    opacityDuration?: number;
    /** Time (ms) for the size animation during exit. Defaults to `exitDuration` for unified timing. */
    exitSizeDuration?: number;
    /** Time (ms) for the opacity animation during exit. Defaults to `exitSizeDuration` for synchronized timing. */
    exitOpacityDuration?: number;
};

declare type CollapseOrientation = 'horizontal' | 'vertical';

export declare type CollapseParams = BasePresenceParams & AnimateOpacity & CollapseDurations & {
    /** The orientation of the size animation. Defaults to `'vertical'` to expand/collapse the height. */
    orientation?: CollapseOrientation;
    /** The starting size for the expand animation. Defaults to `'0px'`. */
    fromSize?: string;
    /**
     * Time (ms) to delay the inner stagger between size and opacity animations.
     * On enter this delays the opacity after size; on exit this delays the size after opacity.
     * Defaults to 0.
     */
    staggerDelay?: number;
    /**
     * Time (ms) to delay the inner stagger during exit. Defaults to the `staggerDelay` param for symmetry.
     */
    exitStaggerDelay?: number;
};

export declare const CollapseRelaxed: PresenceComponent<CollapseParams>;

export declare const CollapseSnappy: PresenceComponent<CollapseParams>;

/** A React component that applies fade in/out transitions to its children. */
export declare const Fade: PresenceComponent<FadeParams>;

export declare type FadeParams = BasePresenceParams & {
    /** The starting opacity value. Defaults to 0. */
    fromOpacity?: number;
    /** The ending opacity value. Defaults to 1. */
    toOpacity?: number;
};

export declare const FadeRelaxed: PresenceComponent<FadeParams>;

export declare const FadeSnappy: PresenceComponent<FadeParams>;

/**
 * Common delay parameters for presence motion components.
 */
declare type PresenceDelay = {
    /** Time (ms) to delay the enter transition. */
    delay?: EffectTiming['delay'];
    /** Time (ms) to delay the exit transition. Defaults to the `delay` param for symmetry. */
    exitDelay?: EffectTiming['delay'];
};

/**
 * Common duration parameters for presence motion components.
 */
declare type PresenceDuration = {
    /** Time (ms) for the enter transition. */
    duration?: number;
    /** Time (ms) for the exit transition. Defaults to the `duration` param for symmetry. */
    exitDuration?: number;
};

/**
 * Common easing parameters for presence motion components.
 */
declare type PresenceEasing = {
    /** Easing curve for the enter transition. */
    easing?: string;
    /** Easing curve for the exit transition. Defaults to the `easing` param for symmetry. */
    exitEasing?: string;
};

export declare const Rotate: PresenceComponent<RotateParams>;

export declare type RotateParams = BasePresenceParams & AnimateOpacity & {
    /**
     * The axis of rotation: 'x', 'y', or 'z'.
     * Defaults to 'z'.
     */
    axis?: Axis3D;
    /**
     * The starting rotation angle in degrees.
     * Defaults to -90.
     */
    fromAngle?: number;
    /**
     * The ending rotation angle in degrees.
     * Defaults to 0.
     */
    toAngle?: number;
};

/** A React component that applies scale in/out transitions to its children. */
export declare const Scale: PresenceComponent<ScaleParams>;

export declare type ScaleParams = BasePresenceParams & AnimateOpacity & {
    /** The scale value to animate from. Defaults to `0.9`. */
    fromScale?: number;
    /** The scale value to animate to. Defaults to `1`. */
    toScale?: number;
};

export declare const ScaleRelaxed: PresenceComponent<ScaleParams>;

export declare const ScaleSnappy: PresenceComponent<ScaleParams>;

/** A React component that applies slide in/out transitions to its children. */
export declare const Slide: PresenceComponent<SlideParams>;

export declare type SlideParams = BasePresenceParams & AnimateOpacity & {
    /** The X translate value with units to animate from. Defaults to `'0px'`. */
    fromX?: string;
    /** The Y translate value with units to animate from. Defaults to `'20px'`. */
    fromY?: string;
    /** The X translate value with units to animate to. Defaults to `'0px'`. */
    toX?: string;
    /** The Y translate value with units to animate to. Defaults to `'0px'`. */
    toY?: string;
};

export declare const SlideRelaxed: PresenceComponent<SlideParams>;

export declare const SlideSnappy: PresenceComponent<SlideParams>;

export { }
