'use client';
import * as React from 'react';
import { useFieldControlProps_unstable } from '@fluentui/react-field';
import { useActiveDescendant } from '@fluentui/react-aria';
import { ChevronDownRegular as ChevronDownIcon, DismissRegular as DismissIcon } from '@fluentui/react-icons';
import { useFluent_unstable as useFluent } from '@fluentui/react-shared-contexts';
import { getPartitionedNativeProps, mergeCallbacks, useMergedRefs, slot, useEventCallback, useOnClickOutside } from '@fluentui/react-utilities';
import { useComboboxBaseState } from '../../utils/useComboboxBaseState';
import { useComboboxPositioning } from '../../utils/useComboboxPositioning';
import { Listbox } from '../Listbox/Listbox';
import { useListboxSlot } from '../../utils/useListboxSlot';
import { useButtonTriggerSlot } from './useButtonTriggerSlot';
import { optionClassNames } from '../Option/useOptionStyles.styles';
/**
 * Create the state required to render Dropdown.
 *
 * The returned state can be modified with hooks such as useDropdownStyles_unstable,
 * before being passed to renderDropdown_unstable.
 *
 * @param props - props from this instance of Dropdown
 * @param ref - reference to root HTMLElement of Dropdown
 */ export const useDropdown_unstable = (props, ref)=>{
    'use no memo';
    var _state_clearButton;
    // Merge props from surrounding <Field>, if any
    props = useFieldControlProps_unstable(props, {
        supportsLabelFor: true,
        supportsSize: true
    });
    const { listboxRef: activeDescendantListboxRef, activeParentRef, controller: activeDescendantController } = useActiveDescendant({
        matchOption: (el)=>el.classList.contains(optionClassNames.root)
    });
    const baseState = useComboboxBaseState({
        ...props,
        activeDescendantController,
        freeform: false
    });
    const { clearable, clearSelection, disabled, hasFocus, multiselect, open, selectedOptions, setOpen } = baseState;
    const { primary: triggerNativeProps, root: rootNativeProps } = getPartitionedNativeProps({
        props,
        primarySlotTagName: 'button',
        excludedPropNames: [
            'children'
        ]
    });
    const [comboboxPopupRef, comboboxTargetRef] = useComboboxPositioning(props);
    const triggerRef = React.useRef(null);
    const listbox = useListboxSlot(props.listbox, useMergedRefs(comboboxPopupRef, activeDescendantListboxRef), {
        state: baseState,
        triggerRef,
        defaultProps: {
            children: props.children
        }
    });
    const { targetDocument } = useFluent();
    useOnClickOutside({
        element: targetDocument,
        callback: (event)=>setOpen(event, false),
        refs: [
            triggerRef,
            comboboxPopupRef,
            comboboxTargetRef
        ],
        disabled: !open
    });
    var _props_button;
    const trigger = useButtonTriggerSlot((_props_button = props.button) !== null && _props_button !== void 0 ? _props_button : {}, useMergedRefs(triggerRef, activeParentRef, ref), {
        state: baseState,
        defaultProps: {
            type: 'button',
            // tabster navigation breaks if the button is disabled and tabIndex is 0
            tabIndex: triggerNativeProps.disabled ? undefined : 0,
            children: baseState.value || props.placeholder,
            'aria-controls': open ? listbox === null || listbox === void 0 ? void 0 : listbox.id : undefined,
            ...triggerNativeProps
        },
        activeDescendantController
    });
    const rootSlot = slot.always(props.root, {
        defaultProps: {
            'aria-owns': !props.inlinePopup && open ? listbox === null || listbox === void 0 ? void 0 : listbox.id : undefined,
            children: props.children,
            ...rootNativeProps
        },
        elementType: 'div'
    });
    rootSlot.ref = useMergedRefs(rootSlot.ref, comboboxTargetRef);
    const showClearButton = selectedOptions.length > 0 && !disabled && clearable && !multiselect;
    const state = {
        components: {
            root: 'div',
            button: 'button',
            clearButton: 'button',
            expandIcon: 'span',
            listbox: Listbox
        },
        root: rootSlot,
        button: trigger,
        listbox: open || hasFocus ? listbox : undefined,
        clearButton: slot.optional(props.clearButton, {
            defaultProps: {
                'aria-label': 'Clear selection',
                children: /*#__PURE__*/ React.createElement(DismissIcon, null),
                // Safari doesn't allow to focus an element with this
                // when the element is not visible (display: none) we need to remove it to avoid tabster issues
                tabIndex: showClearButton ? 0 : undefined,
                type: 'button'
            },
            elementType: 'button',
            renderByDefault: true
        }),
        expandIcon: slot.optional(props.expandIcon, {
            renderByDefault: true,
            defaultProps: {
                children: /*#__PURE__*/ React.createElement(ChevronDownIcon, null)
            },
            elementType: 'span'
        }),
        placeholderVisible: !baseState.value && !!props.placeholder,
        showClearButton,
        activeDescendantController,
        ...baseState
    };
    const onClearButtonClick = useEventCallback(mergeCallbacks((_state_clearButton = state.clearButton) === null || _state_clearButton === void 0 ? void 0 : _state_clearButton.onClick, (ev)=>{
        var _triggerRef_current;
        clearSelection(ev);
        (_triggerRef_current = triggerRef.current) === null || _triggerRef_current === void 0 ? void 0 : _triggerRef_current.focus();
    }));
    if (state.clearButton) {
        state.clearButton.onClick = onClearButtonClick;
    }
    // Heads up! We don't support "clearable" in multiselect mode, so we should never display a slot
    if (multiselect) {
        state.clearButton = undefined;
    }
    if (process.env.NODE_ENV !== 'production') {
        // eslint-disable-next-line react-hooks/rules-of-hooks -- "process.env" does not change in runtime
        React.useEffect(()=>{
            if (clearable && multiselect) {
                // eslint-disable-next-line no-console
                console.error(`[@fluentui/react-combobox] "clearable" prop is not supported in multiselect mode.`);
            }
        }, [
            clearable,
            multiselect
        ]);
    }
    return state;
};
