import { ComponentProps } from '@fluentui/react-utilities';
import { ComponentState } from '@fluentui/react-utilities';
import type { ForwardRefComponent } from '@fluentui/react-utilities';
import type { JSXElement } from '@fluentui/react-utilities';
import { Label } from '@fluentui/react-label';
import * as React_2 from 'react';
import { Slot } from '@fluentui/react-utilities';
import type { SlotClassNames } from '@fluentui/react-utilities';

/**
 * Checkboxes give people a way to select one or more items from a group,
 * or switch between two mutually exclusive options (checked or unchecked).
 */
export declare const Checkbox: ForwardRefComponent<CheckboxProps>;

export declare const checkboxClassNames: SlotClassNames<CheckboxSlots>;

/**
 * Data for the onChange event for checkbox.
 */
export declare interface CheckboxOnChangeData {
    checked: 'mixed' | boolean;
}

/**
 * Checkbox Props
 */
export declare type CheckboxProps = Omit<ComponentProps<Partial<CheckboxSlots>, 'input'>, 'checked' | 'defaultChecked' | 'onChange' | 'size'> & {
    /**
     * The controlled value for the checkbox.
     *
     * @default false
     */
    checked?: 'mixed' | boolean;
    /**
     * Checkboxes don't support children. To add a label, use the `label` prop.
     */
    children?: never;
    /**
     * Whether the checkbox should be rendered as checked by default.
     */
    defaultChecked?: 'mixed' | boolean;
    /**
     * The position of the label relative to the checkbox indicator.
     *
     * @default after
     */
    labelPosition?: 'before' | 'after';
    /**
     * Callback to be called when the checked state value changes.
     */
    onChange?: (ev: React_2.ChangeEvent<HTMLInputElement>, data: CheckboxOnChangeData) => void;
    /**
     * The shape of the checkbox indicator.
     *
     * The `circular` variant is only recommended to be used in a tasks-style UI (checklist),
     * since it otherwise could be confused for a `RadioItem`.
     *
     * @default square
     */
    shape?: 'square' | 'circular';
    /**
     * The size of the checkbox indicator.
     *
     * @default medium
     */
    size?: 'medium' | 'large';
};

export declare type CheckboxSlots = {
    /**
     * The root element of the Checkbox.
     *
     * The root slot receives the `className` and `style` specified directly on the `<Checkbox>`.
     * All other native props will be applied to the primary slot: `input`
     */
    root: NonNullable<Slot<'span'>>;
    /**
     * The Checkbox's label.
     */
    label?: Slot<typeof Label>;
    /**
     * Hidden input that handles the checkbox's functionality.
     *
     * This is the PRIMARY slot: all native properties specified directly on `<Checkbox>` will be applied to this slot,
     * except `className` and `style`, which remain on the root slot.
     */
    input: NonNullable<Slot<'input'>>;
    /**
     * The checkbox, with the checkmark icon as its child when checked.
     */
    indicator: Slot<'div', 'span'>;
};

/**
 * State used in rendering Checkbox
 */
export declare type CheckboxState = ComponentState<CheckboxSlots> & Required<Pick<CheckboxProps, 'checked' | 'disabled' | 'labelPosition' | 'shape' | 'size'>>;

export declare const renderCheckbox_unstable: (state: CheckboxState) => JSXElement;

/**
 * Create the state required to render Checkbox.
 *
 * The returned state can be modified with hooks such as useCheckboxStyles_unstable,
 * before being passed to renderCheckbox_unstable.
 *
 * @param props - props from this instance of Checkbox
 * @param ref - reference to `<input>` element of Checkbox
 */
export declare const useCheckbox_unstable: (props: CheckboxProps, ref: React_2.Ref<HTMLInputElement>) => CheckboxState;

/**
 * Apply styling to the Checkbox slots based on the state
 */
export declare const useCheckboxStyles_unstable: (state: CheckboxState) => CheckboxState;

export { }
