import type { ARIAButtonSlotProps } from '@fluentui/react-aria';
import type { ComponentProps } from '@fluentui/react-utilities';
import type { ComponentState } from '@fluentui/react-utilities';
import { ForwardRefComponent } from '@fluentui/react-utilities';
import type { JSXElement } from '@fluentui/react-utilities';
import * as React_2 from 'react';
import type { Slot } from '@fluentui/react-utilities';
import type { SlotClassNames } from '@fluentui/react-utilities';

/**
 * Buttons give people a way to trigger an action.
 */
export declare const Button: ForwardRefComponent<ButtonProps>;

export declare const buttonClassNames: SlotClassNames<ButtonSlots>;

/**
 * @internal
 * Internal context provider used to update default values between internal components
 */
export declare const ButtonContextProvider: React_2.Provider<ButtonContextValue | undefined>;

/**
 * @internal
 * Internal context value used to update default values between internal components
 */
export declare interface ButtonContextValue {
    size?: ButtonSize;
}

export declare type ButtonProps = ComponentProps<ButtonSlots> & {
    /**
     * A button can have its content and borders styled for greater emphasis or to be subtle.
     * - 'secondary' (default): Gives emphasis to the button in such a way that it indicates a secondary action.
     * - 'primary': Emphasizes the button as a primary action.
     * - 'outline': Removes background styling.
     * - 'subtle': Minimizes emphasis to blend into the background until hovered or focused.
     * - 'transparent': Removes background and border styling.
     *
     * @default 'secondary'
     */
    appearance?: 'secondary' | 'primary' | 'outline' | 'subtle' | 'transparent';
    /**
     * When set, allows the button to be focusable even when it has been disabled. This is used in scenarios where it
     * is important to keep a consistent tab order for screen reader and keyboard users. The primary example of this
     * pattern is when the disabled button is in a menu or a commandbar and is seldom used for standalone buttons.
     *
     * @default false
     */
    disabledFocusable?: boolean;
    /**
     * A button can show that it cannot be interacted with.
     *
     * @default false
     */
    disabled?: boolean;
    /**
     * A button can format its icon to appear before or after its content.
     *
     * @default 'before'
     */
    iconPosition?: 'before' | 'after';
    /**
     * A button can be rounded, circular, or square.
     *
     * @default 'rounded'
     */
    shape?: 'rounded' | 'circular' | 'square';
    /**
     * A button supports different sizes.
     *
     * @default 'medium'
     */
    size?: ButtonSize;
};

/**
 * A button supports different sizes.
 */
declare type ButtonSize = 'small' | 'medium' | 'large';

export declare type ButtonSlots = {
    /**
     * Root of the component that renders as either a `<button>` tag or an `<a>` tag.
     */
    root: NonNullable<Slot<ARIAButtonSlotProps<'a'>>>;
    /**
     * Icon that renders either before or after the `children` as specified by the `iconPosition` prop.
     */
    icon?: Slot<'span'>;
};

export declare type ButtonState = ComponentState<ButtonSlots> & Required<Pick<ButtonProps, 'appearance' | 'disabledFocusable' | 'disabled' | 'iconPosition' | 'shape' | 'size'>> & {
    /**
     * A button can contain only an icon.
     *
     * @default false
     */
    iconOnly: boolean;
};

/**
 * CompoundButtons are buttons that can have secondary content that adds extra information to the user.
 */
export declare const CompoundButton: ForwardRefComponent<CompoundButtonProps>;

export declare const compoundButtonClassNames: SlotClassNames<CompoundButtonSlots>;

export declare type CompoundButtonProps = ComponentProps<Partial<CompoundButtonSlots>> & Pick<ButtonProps, 'appearance' | 'disabledFocusable' | 'disabled' | 'iconPosition' | 'shape' | 'size'>;

export declare type CompoundButtonSlots = ButtonSlots & {
    /**
     * Second line of text that describes the action this button takes.
     */
    secondaryContent?: Slot<'span'>;
    /**
     * Container that wraps the children and the secondaryContent slot.
     */
    contentContainer: NonNullable<Slot<'span'>>;
};

export declare type CompoundButtonState = ComponentState<CompoundButtonSlots> & Omit<ButtonState, keyof ButtonSlots | 'components'>;

/**
 * MenuButtons are buttons that have a chevron icon after the button contents and are usually clicked to open/close
 * menus.
 */
export declare const MenuButton: ForwardRefComponent<MenuButtonProps>;

export declare const menuButtonClassNames: SlotClassNames<MenuButtonSlots>;

export declare type MenuButtonProps = ComponentProps<MenuButtonSlots> & Pick<ButtonProps, 'appearance' | 'disabledFocusable' | 'disabled' | 'shape' | 'size'>;

export declare type MenuButtonSlots = ButtonSlots & {
    /**
     * Menu icon that indicates that this button has a menu that can be expanded.
     */
    menuIcon?: Slot<'span'>;
};

export declare type MenuButtonState = ComponentState<MenuButtonSlots> & Omit<ButtonState, keyof ButtonSlots | 'components' | 'iconPosition'>;

/**
 * Renders a Button component by passing the state defined props to the appropriate slots.
 */
declare const renderButton_unstable: (state: ButtonState) => JSXElement;
export { renderButton_unstable }
export { renderButton_unstable as renderToggleButton_unstable }

/**
 * Renders a CompoundButton component by passing the state defined props to the appropriate slots.
 */
export declare const renderCompoundButton_unstable: (state: CompoundButtonState) => JSXElement;

/**
 * Renders a MenuButton component by passing the state defined props to the appropriate slots.
 */
export declare const renderMenuButton_unstable: (state: MenuButtonState) => JSXElement;

/**
 * Renders a SplitButton component by passing the state defined props to the appropriate slots.
 */
export declare const renderSplitButton_unstable: (state: SplitButtonState) => JSXElement;

/**
 * SplitButtons are a grouping of two interactive surfaces where interacting with the first one triggers a primary
 * action, while interacting with the second one opens a menu with secondary actions.
 */
export declare const SplitButton: ForwardRefComponent<SplitButtonProps>;

export declare const splitButtonClassNames: SlotClassNames<SplitButtonSlots>;

export declare type SplitButtonProps = ComponentProps<SplitButtonSlots> & Omit<ButtonProps, 'root' | 'as'> & Omit<MenuButtonProps, 'root' | 'as'>;

export declare type SplitButtonSlots = {
    /**
     * Root of the component that wraps the primary action button and menu button.
     */
    root: NonNullable<Slot<'div'>>;
    /**
     * Button that opens menu with secondary actions in SplitButton.
     */
    menuButton?: Slot<typeof MenuButton>;
    /**
     * Button to perform primary action in SplitButton.
     */
    primaryActionButton?: Slot<typeof Button>;
};

export declare type SplitButtonState = ComponentState<SplitButtonSlots> & Omit<ButtonState, 'components' | 'iconOnly' | 'root'> & Omit<MenuButtonState, 'components' | 'iconOnly' | 'root'>;

/**
 * ToggleButtons are buttons that toggle between two defined states when triggered.
 */
export declare const ToggleButton: ForwardRefComponent<ToggleButtonProps>;

export declare const toggleButtonClassNames: SlotClassNames<ButtonSlots>;

export declare type ToggleButtonProps = ButtonProps & {
    /**
     * Defines whether the `ToggleButton` is initially in a checked state or not when rendered.
     *
     * @default false
     */
    defaultChecked?: boolean;
    /**
     * Defines the controlled checked state of the `ToggleButton`.
     * If passed, `ToggleButton` ignores the `defaultChecked` property.
     * This should only be used if the checked state is to be controlled at a higher level and there is a plan to pass the
     * correct value based on handling `onClick` events and re-rendering.
     *
     * @default false
     */
    checked?: boolean;
};

export declare type ToggleButtonState = ButtonState & Required<Pick<ToggleButtonProps, 'checked'>>;

/**
 * Given user props, defines default props for the Button, calls useButtonState, and returns processed state.
 * @param props - User provided props to the Button component.
 * @param ref - User provided ref to be passed to the Button component.
 */
export declare const useButton_unstable: (props: ButtonProps, ref: React_2.Ref<HTMLButtonElement | HTMLAnchorElement>) => ButtonState;

/**
 * @internal
 * Internal context hook used to update default values between internal components
 */
export declare const useButtonContext: () => ButtonContextValue;

export declare const useButtonStyles_unstable: (state: ButtonState) => ButtonState;

/**
 * Given user props, defines default props for the CompoundButton, calls useButtonState, and returns processed state.
 * @param props - User provided props to the CompoundButton component.
 * @param ref - User provided ref to be passed to the CompoundButton component.
 */
export declare const useCompoundButton_unstable: ({ contentContainer, secondaryContent, ...props }: CompoundButtonProps, ref: React_2.Ref<HTMLButtonElement | HTMLAnchorElement>) => CompoundButtonState;

export declare const useCompoundButtonStyles_unstable: (state: CompoundButtonState) => CompoundButtonState;

/**
 * Given user props, returns the final state for a MenuButton.
 */
export declare const useMenuButton_unstable: ({ menuIcon, ...props }: MenuButtonProps, ref: React_2.Ref<HTMLButtonElement | HTMLAnchorElement>) => MenuButtonState;

export declare const useMenuButtonStyles_unstable: (state: MenuButtonState) => MenuButtonState;

/**
 * Given user props, defines default props for the SplitButton and returns processed state.
 * @param props - User provided props to the SplitButton component.
 * @param ref - User provided ref to be passed to the SplitButton component.
 */
export declare const useSplitButton_unstable: (props: SplitButtonProps, ref: React_2.Ref<HTMLButtonElement | HTMLAnchorElement>) => SplitButtonState;

export declare const useSplitButtonStyles_unstable: (state: SplitButtonState) => SplitButtonState;

/**
 * Given user props, defines default props for the ToggleButton, calls useButtonState and useChecked, and returns
 * processed state.
 * @param props - User provided props to the ToggleButton component.
 * @param ref - User provided ref to be passed to the ToggleButton component.
 */
export declare const useToggleButton_unstable: (props: ToggleButtonProps, ref: React_2.Ref<HTMLButtonElement | HTMLAnchorElement>) => ToggleButtonState;

export declare const useToggleButtonStyles_unstable: (state: ToggleButtonState) => ToggleButtonState;

export declare function useToggleState<TToggleButtonProps extends Pick<ToggleButtonProps, 'checked' | 'defaultChecked' | 'disabled' | 'disabledFocusable'>, TButtonState extends Pick<ButtonState, 'root'>, TToggleButtonState extends Pick<ToggleButtonState, 'checked' | 'root'>>(props: TToggleButtonProps, state: TButtonState): TToggleButtonState;

export { }
